
/*
    RFC - String.h
    Copyright (C) 2010 Ruchira Hasaranga
  
    This software is provided 'as-is', without any express or implied
    warranty.  In no event will the authors be held liable for any damages
    arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute it
    freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you must not
       claim that you wrote the original software. If you use this software
       in a product, an acknowledgment in the product documentation would be
       appreciated but is not required.
    2. Altered source versions must be plainly marked as such, and must not be
       misrepresented as being the original software.
    3. This notice may not be removed or altered from any source distribution.
      
*/

#ifndef _RFCSTRING_H_
#define _RFCSTRING_H_

#define _CRT_SECURE_NO_DEPRECATE
#pragma warning(disable: 4996) // ignore depricated warning!

#include"../Object.h"
#include<windows.h>
#include<string.h>
#include<malloc.h>

/**
	Simple string class which has unicode support! 
	But not support multiple zero terminated strings.
*/
class String : public Object
{
protected:
	char *a_text;
	wchar_t *w_text;
	int count; // character count!

public:

	/**
		Constructs an empty string
	*/
	String()
	{
		a_text=0;
		w_text=0;
		count=0;
	}


	/**
		Constructs copy of another string
	*/
	String(const String& other)
	{
		if(other.count)
		{
			a_text=(char*)malloc(other.count+1);
			strcpy(a_text,other.a_text);

			w_text=(wchar_t*)malloc((other.count+1)*sizeof(wchar_t));
			wcscpy(w_text,other.w_text);
		}else
		{
			a_text=0;
			w_text=0;
		}

		count=other.count;		
	}

	/**
		Constructs String object using ansi string
	*/
	String(const char* const text,UINT codePage=CP_ACP)
	{
		if ( (text != 0) && (*text != 0) )
		{
			int length = (int)strlen(text);
			a_text=(char*)malloc(length+1);
			strcpy(a_text,text);

			count=length;

			length = MultiByteToWideChar(codePage, 0, text, -1, 0,0);
			if (length)
			{
				w_text = (wchar_t *)malloc(length*sizeof(wchar_t));
				if (MultiByteToWideChar(codePage, 0, text, -1, w_text,length))
				{
					return;
				}else
				{
					free(w_text);
					free(a_text);
				}
			}else
			{
				free(a_text);
			}
		}

		a_text=0;
		w_text=0;
		count=0;		
	}

	/**
		Constructs String object using unicode string
	*/
	String(const wchar_t* const text,UINT codePage=CP_ACP)
	{
		if ( (text != 0) && (*text != 0) )
		{
			int length = (int)wcslen(text);
			w_text=(wchar_t*)malloc((length+1)*sizeof(wchar_t));
			wcscpy(w_text,text);

			count=length;

			length = WideCharToMultiByte(codePage,0, text, -1, 0,0,0,0);
			if (length)
			{
				a_text = (char*)malloc(length);
				if (WideCharToMultiByte(codePage,0, text, -1, a_text,length,0,0))
				{
					return;
				}else
				{
					free(w_text);
					free(a_text);
				}
			}else
			{
				free(w_text);
			}
		}

		a_text=0;
		w_text=0;
		count=0;		
	}

	/**
		Constructs String object using integer
	*/
	String(const int value,const int radix)
	{
		char text[33];
		_itoa(value,text,radix);
		
		int length = (int)strlen(text);
		a_text=(char*)malloc(length+1);
		strcpy(a_text,text);

		count=length;

		length = MultiByteToWideChar(CP_ACP, 0, text, -1, 0,0);
		if (length)
		{
			w_text = (wchar_t *)malloc(length*sizeof(wchar_t));
			if (!MultiByteToWideChar(CP_ACP, 0, text, -1, w_text,length))			
			{
				free(w_text);
				free(a_text);
				a_text=0;
				w_text=0;
				count=0;
			}
		}else
		{
			free(a_text);
			a_text=0;
			w_text=0;
			count=0;
		}
	}

	/** 
		Replaces this string's contents with another string.
	*/
	const String& operator= (const String& other)
	{
		if(other.count)
		{
			if(count)
			{
				free(a_text);
				free(w_text);
			}
			a_text=(char*)malloc(other.count+1);
			w_text=(wchar_t*)malloc((other.count+1)*sizeof(wchar_t));
			strcpy(a_text,other.a_text);
			wcscpy(w_text,other.w_text);
			count=other.count;
		}else
		{
			if(count)
			{
				free(a_text);
				free(w_text);
				a_text=0;
				w_text=0;
				count=0;
			}

		}
		return *this;
	}

	/** 
		Replaces this string's contents with unicode string. 
	*/
	const String& operator= (const wchar_t* const other)
	{
		int length = (int)wcslen(other);
		if(length)
		{
			if(count)
			{
				free(a_text);
				free(w_text);
			}

			w_text=(wchar_t*)malloc((length+1)*sizeof(wchar_t));
			wcscpy(w_text,other);

			count=length;

			length = WideCharToMultiByte(CP_ACP,0, other, -1, 0,0,0,0);
			if (length)
			{
				a_text = (char*)malloc(length);
				if (WideCharToMultiByte(CP_ACP,0, other, -1, a_text,length,0,0))
				{
					return *this;
				}else
				{
					free(a_text);
					free(w_text);
				}
			}else
			{
				free(w_text);
			}
		}else
		{
			if(count)
			{
				free(a_text);
				free(w_text);
			}
		}

		a_text=0;
		w_text=0;
		count=0;

		return *this;
	}


    /** Appends a string at the end of this one.
        @returns     the concatenated string
    */
    const String operator+ (const String& stringToAppend)
	{
		return Append(stringToAppend);
	}

    /** Appends a unicode string at the end of this one.
        @returns     the concatenated string
    */
	const String operator+ (const wchar_t* const textToAppend)
	{
		return Append(String(textToAppend));
	}

	/**
		Returns ansi version of this string
	*/
	operator const char*()
	{
		return a_text;
	}

	/**
		Returns unicode version of this string
	*/
	operator const wchar_t*()
	{
		return w_text;
	}

    /** 
		Returns a character from the string.
		@returns -1 if index is out of range
    */
	const char operator[](const int index)
	{
		if((0<=index) && (index<=(count-1)))
			return a_text[index];
		return -1;
	}

	/**
	    Appends a string at the end of this one.
        @returns     the concatenated string
	*/
	virtual String Append(const String &otherString)
	{
		if(otherString.count)
		{
			String result;
			int length=otherString.count;
			result.w_text=(wchar_t*)malloc((length+count+1)*sizeof(wchar_t));
			result.a_text=(char*)malloc(length+count+1);
			if(count) // this string is not empty!
			{
				wcscpy(result.w_text,w_text);
				wcsncat(result.w_text,otherString.w_text,length);

				strcpy(result.a_text,a_text);
				strncat(result.a_text,otherString.a_text,length);
			}else
			{
				wcscpy(result.w_text,otherString.w_text);
				strcpy(result.a_text,otherString.a_text);
			}

			result.count=length+count;
			return result;
		}else
		{
			return *this;
		}
	}

    /** 
		Returns a subsection of the string.

        If the range specified is beyond the limits of the string, empty string
        will be return.

        @param start   the index of the start of the substring needed
        @param end     all characters from start up to this index are returned
    */
	virtual String SubString(int start,int end)
	{
		if((0<=start) && (start<=(count-1)))
		{
			if((start<end) && (end<=(count-1)))
			{
				int size=(end-start)+1;
				wchar_t* buf=(wchar_t*)malloc((size+1)*sizeof(wchar_t));
				wcsncpy(buf,&w_text[start],size);
				buf[size]=0;

				String result(buf);
				free(buf);
				return result;
			}
		}
		return String();
	}

    /** 
		Case-insensitive comparison with another string.
        @returns     true if the two strings are identical, false if not
    */
	virtual bool EqualsIgnoreCase(String &otherString)
	{
		if(wcscmp(w_text,otherString.w_text)==0)
			return true;
		return false;
	}

	/**
		Compare first character with given unicode character
	*/
	virtual bool StartsWithChar(wchar_t character)
	{
		if(w_text[0]==character)
			return true;
		return false;
	}

	/**
		Compare first character with given ansi character
	*/
	virtual bool StartsWithChar(char character)
	{
		if(a_text[0]==character)
			return true;
		return false;
	}

	/**
		Compare last character with given unicode character
	*/
	virtual bool EndsWithChar(wchar_t character)
	{
		if(w_text[count-1]==character)
			return true;
		return false;
	}

	/**
		Compare last character with given ansi character
	*/
	virtual bool EndsWithChar(char character)
	{
		if(a_text[count-1]==character)
			return true;
		return false;
	}

	/**
		Check if string is quoted or not
	*/
	virtual bool IsQuotedString()
	{
		if(count>1)
		{
			if(StartsWithChar(L'\"') && EndsWithChar(L'\"'))
				return true;
		}
		return false;
	}

    /** 
		Returns a character from the string.
		@returns -1 if index is out of range
    */
	virtual wchar_t GetCharAt(int index)
	{
		if((0<=index) && (index<=(count-1)))
			return w_text[index];
		return -1;
	}

	/**
		Returns number of characters in string
	*/
	virtual int GetLength()
	{
		return count;
	}

	/**
		Returns true if string is empty
	*/
	virtual bool IsEmpty()
	{
		if(count)
			return false;
		return true;
	}

	/**
		Returns value of string
	*/
	virtual int GetIntValue()
	{
		if(IsEmpty())
			return 0;
		return _wtoi(w_text);
	}

	virtual ~String()
	{
		if(count)
		{
			free(a_text);
			free(w_text);
		}
	}
};

const String operator+ (const char* const string1,const String& string2);

const String operator+ (const wchar_t* const string1,const String& string2);

const String operator+ (const String& string1,const String& string2);

#endif