

// Portable-ANSI JS Support for RCOM
// (c) 2016 - 2018 R.Hasaranga

// this is a 32bit code because of using int for reference type

/*
	make sure to own js objects/values while changing contexts! otherwise gc will f#ckup.
	ex:	
		v7_val_t obj = v7_mk_object(v7);
		v7_own(v7, &obj);
		.
		.
		.
		call js code
		.
		.
		.
		v7_disown(v7, &obj);
*/

/* build flags:
V7_BUILD_PROFILE=2
V7_LARGE_AST
V7_ENABLE_CRYPTO
V7_ENABLE_FILE
V7_ENABLE_SOCKET
V7_ENABLE__Math
V7_ENABLE__Math__abs
V7_ENABLE__Math__acos
V7_ENABLE__Math__asin
V7_ENABLE__Math__atan
V7_ENABLE__Math__atan2
V7_ENABLE__Math__ceil
V7_ENABLE__Math__cos
V7_ENABLE__Math__exp
V7_ENABLE__Math__floor
V7_ENABLE__Math__log
V7_ENABLE__Math__max
V7_ENABLE__Math__min
V7_ENABLE__Math__pow
V7_ENABLE__Math__random
V7_ENABLE__Math__round
V7_ENABLE__Math__sin
V7_ENABLE__Math__sqrt
V7_ENABLE__Math__tan
*/

#include"KPointerList.h"
#include"../rcom.h"
#include"v7.h"
#include <string.h>

#ifndef WIN32 // for linux

#include <uuid/uuid.h>
#include <dirent.h>
#include <unistd.h>
#include <sys/stat.h>

#define RPC_S_OK 0
#define UUID GUID

void ByteSwapGUID(GUID *guid)
{
	// byte swapping = little endian <-> big endian

	guid->Data1 = __builtin_bswap32(guid->Data1);
	guid->Data2 = ( (guid->Data2 & 0xFF) << 8 ) | ( (guid->Data2 & 0xFF00) >> 8 );
	guid->Data3 = ( (guid->Data3 & 0xFF) << 8 ) | ( (guid->Data3 & 0xFF00) >> 8 );
}

int UuidFromStringA(char* text, GUID *guid)
{
	int retVal = uuid_parse(text, (unsigned char*)guid);
	ByteSwapGUID(guid); // convert to microsoft guid
	return retVal;
}

void UuidToStringA(GUID* guid, char** text)
{
	GUID linGUID;
	memcpy(&linGUID, guid, sizeof(GUID)); // make a copy of it

	ByteSwapGUID(&linGUID); // convert to linux uuid

	char *buffer = (char*)malloc(37);
	uuid_unparse((unsigned char*)&linGUID, buffer);
	*text = buffer;
}

void RpcStringFreeA(char** text)
{
	free(*text);
}

#endif

// {C0461FBF-DCD4-45ef-8F2F-C6D9E808E133}
static const GUID CLSID_JSFactory_Default = { 0xc0461fbf, 0xdcd4, 0x45ef, { 0x8f, 0x2f, 0xc6, 0xd9, 0xe8, 0x8, 0xe1, 0x33 } };


RCOMSystemAPI *systemAPI;
char pluginPath[512];
char jsPluginPath[512];
KPointerList<char*> jsCompList;

static enum v7_err js_Alert(struct v7 *v7, v7_val_t *res);
static enum v7_err js_JSObjectToRCOMObject(struct v7 *v7, v7_val_t *res);
static enum v7_err js_ToStringArray(struct v7 *v7, v7_val_t *res);
static enum v7_err js_ToIntArray(struct v7 *v7, v7_val_t *res);
static enum v7_err js_ToFloatArray(struct v7 *v7, v7_val_t *res);
static enum v7_err js_ToDoubleArray(struct v7 *v7, v7_val_t *res);
static enum v7_err js_ToBoolArray(struct v7 *v7, v7_val_t *res);
static enum v7_err js_ToRCOMObjectArray(struct v7 *v7, v7_val_t *res);
static enum v7_err js_Print(struct v7 *v7, v7_val_t *res);
static enum v7_err js_Inlcude(struct v7 *v7, v7_val_t *res);


static enum v7_err api_IsFactoryAvailable(struct v7 *v7, v7_val_t *res);
static enum v7_err api_GetObjectFromFactory(struct v7 *v7, v7_val_t *res);
static enum v7_err api_GetApplicationDir(struct v7 *v7, v7_val_t *res);
static enum v7_err api_GetCommandlineArgCount(struct v7 *v7, v7_val_t *res);
static enum v7_err api_GetCommandlineArgument(struct v7 *v7, v7_val_t *res);
static enum v7_err api_SetNewInstanceListener(struct v7 *v7, v7_val_t *res);
static enum v7_err api_GetNewInstanceListener(struct v7 *v7, v7_val_t *res);
static enum v7_err api_ShutdownSystem(struct v7 *v7, v7_val_t *res);

v7_val_t ConvertRCOMObjectToJS(struct v7 *v7, RCOMObject *object);

static enum v7_err rcom_AddRef(struct v7 *v7, v7_val_t *res);
static enum v7_err rcom_Release(struct v7 *v7, v7_val_t *res);	
static enum v7_err rcom_GetTypeInfo(struct v7 *v7, v7_val_t *res);
static enum v7_err rcom_Invoke(struct v7 *v7, v7_val_t *res);
static enum v7_err rcom_OverrideInvoke(struct v7 *v7, v7_val_t *res);
static enum v7_err rcom_SetParent(struct v7 *v7, v7_val_t *res);

class JSFactory : public RCOMFactory
{
protected:
	long refCount;
	GUID factoryID;

	char *fileName;
	struct v7 *v7;
	v7_val_t cobj_Factory;
	v7_val_t scriptRCOMFactory;
	v7_val_t func_Init,func_Exec,func_QueryObject,func_Stop,func_Quit;
	
public:
	bool scriptError;

	void AddGlobals()
	{
		v7_set_method(v7, v7_get_global(v7), "alert", &js_Alert);
		v7_set_method(v7, v7_get_global(v7), "convertToRCOMObject", &js_JSObjectToRCOMObject); 
		v7_set_method(v7, v7_get_global(v7), "toStringArray", &js_ToStringArray);
		v7_set_method(v7, v7_get_global(v7), "toIntArray", &js_ToIntArray);
		v7_set_method(v7, v7_get_global(v7), "toFloatArray", &js_ToFloatArray);
		v7_set_method(v7, v7_get_global(v7), "toDoubleArray", &js_ToDoubleArray);
		v7_set_method(v7, v7_get_global(v7), "toBoolArray", &js_ToBoolArray); 
		v7_set_method(v7, v7_get_global(v7), "toRCOMObjectArray", &js_ToRCOMObjectArray); 
		v7_set_method(v7, v7_get_global(v7), "print", &js_Print); 
		v7_set_method(v7, v7_get_global(v7), "include", &js_Inlcude);
	}

	JSFactory(const char *jsFactoryFile)
	{
		refCount=1;
		scriptError=false;
		fileName=strdup(jsFactoryFile);

		v7 = v7_create();

		// add "this" ptr as user data
		cobj_Factory=v7_mk_object(v7);
		v7_own(v7, &cobj_Factory);
		v7_set_user_data(v7, cobj_Factory, this);
		v7_set(v7, v7_get_global(v7), "cobj_Factory", 12, cobj_Factory);

		AddGlobals();

 		v7_val_t exec_result;
		enum v7_err err=v7_exec_file(v7, jsFactoryFile, &exec_result);

		if(err==V7_OK)
		{
			GetFactoryObject();
		}else
		{
			scriptError=true;
			AddToLog(v7_get_parser_error(v7),"[script error]");
		}
	}

	void GetFactoryObject()
	{
		v7_val_t func_GetRCOMFactory, func_GetFactoryID, result, args;

		func_GetRCOMFactory = v7_get(v7, v7_get_global(v7), "GetRCOMFactory", 14);

		if (func_GetRCOMFactory == V7_UNDEFINED)
		{
			scriptError = true;
			AddToLog("cannot find \"GetRCOMFactory\" function");
			return;
		}

		args = v7_mk_array(v7); // empty params

		if (v7_apply(v7, func_GetRCOMFactory, V7_UNDEFINED, args, &scriptRCOMFactory) == V7_OK) // call GetRCOMFactory
		{
			if (v7_is_object(scriptRCOMFactory)) // returned value is an object
			{
				v7_own(v7, &scriptRCOMFactory);
				func_GetFactoryID = v7_get(v7, scriptRCOMFactory, "GetFactoryID", 12); // query for GetFactoryID method in that obj
				if (func_GetFactoryID != V7_UNDEFINED)
				{
					args = v7_mk_array(v7); // empty params
					if (v7_apply(v7, func_GetFactoryID, scriptRCOMFactory, args, &result) == V7_OK) // call GetFactoryID
					{
						const char *strGUID = v7_get_cstring(v7, &result);
						if (strGUID != NULL)
						{
							// convert guid string to guid
							if (UuidFromStringA((char*)strGUID, &factoryID) == RPC_S_OK)
							{
								// query for other methods
								func_Init = v7_get(v7, scriptRCOMFactory, "Init", 4); // query for Init method
								if (func_Init != V7_UNDEFINED)
								{
									func_Exec = v7_get(v7, scriptRCOMFactory, "Exec", 4); // query for Exec method
									if (func_Exec != V7_UNDEFINED)
									{
										func_QueryObject = v7_get(v7, scriptRCOMFactory, "QueryObject", 11); // query for QueryObject method
										if (func_QueryObject != V7_UNDEFINED)
										{
											func_Stop = v7_get(v7, scriptRCOMFactory, "Stop", 4); // query for Stop method
											if (func_Stop != V7_UNDEFINED)
											{
												func_Quit = v7_get(v7, scriptRCOMFactory, "Quit", 4); // query for Quit method
												if (func_Quit != V7_UNDEFINED)
												{
													return; // everything is ok
												}
												else
												{
													AddToLog("cannot find \"Quit\" method in Factory object");
												}
											}
											else
											{
												AddToLog("cannot find \"Stop\" method in Factory object");
											}
										}
										else
										{
											AddToLog("cannot find \"QueryObject\" method in Factory object");
										}
									}
									else
									{
										AddToLog("cannot find \"Exec\" method in Factory object");
									}
								}
								else
								{
									AddToLog("cannot find \"Init\" method in Factory object");
								}
							}
							else
							{
								AddToLog("invalid guid returned by \"GetFactoryID\" method of Factory object");
							}
						}
						else
						{
							AddToLog("empty guid returned by \"GetFactoryID\" method of Factory object");
						}
					}
					else
					{
						AddToLog(v7_get_parser_error(v7), "execute error, \"GetFactoryID\" method of Factory object");
					}
				}
				else
				{
					AddToLog("cannot find \"GetFactoryID\" method in Factory object");
				}
			}
			else
			{
				AddToLog("invalid object returned from \"GetRCOMFactory\" function");
			}
		}
		else
		{
			AddToLog(v7_get_parser_error(v7),"execute error, \"GetRCOMFactory\" function");
		}
		scriptError = true;

	}

	void AddToLog(const char *text,const char *title=0)
	{
		if (title)
			fprintf(stderr, "[JS Log] %s {%s}: %s\n", title, fileName, text);
		else
			fprintf(stderr, "[JS Log] {%s}: %s\n", fileName,text);
	}

	~JSFactory()
	{
		v7_disown(v7, &cobj_Factory);
		v7_disown(v7, &scriptRCOMFactory);

		free(fileName);
		v7_destroy(v7);
	}

	long RCALL AddRef()
	{
		ADD_REF_CODE
	}

	long RCALL Release()
	{
		RELEASE_CODE
	}	

	bool RCALL Init(RCOMSystemAPI *_systemAPI,bool isFinal)
	{
		if(scriptError)
			return false;

		systemAPI = _systemAPI;

		v7_val_t obj_SystemAPI = v7_mk_object(v7);
		v7_set_method(v7, obj_SystemAPI, "ShutdownSystem", &api_ShutdownSystem);		
		v7_set_method(v7, obj_SystemAPI, "IsFactoryAvailable", &api_IsFactoryAvailable);		
		v7_set_method(v7, obj_SystemAPI, "GetObjectFromFactory", &api_GetObjectFromFactory);		
		v7_set_method(v7, obj_SystemAPI, "GetApplicationDir", &api_GetApplicationDir);		
		v7_set_method(v7, obj_SystemAPI, "GetCommandlineArgCount", &api_GetCommandlineArgCount);		
		v7_set_method(v7, obj_SystemAPI, "GetCommandlineArgument", &api_GetCommandlineArgument);		
		v7_set_method(v7, obj_SystemAPI, "SetNewInstanceListener", &api_SetNewInstanceListener);		
		v7_set_method(v7, obj_SystemAPI, "GetNewInstanceListener", &api_GetNewInstanceListener);

		v7_val_t result, args;
		args = v7_mk_array(v7); // params
		v7_array_push(v7, args, obj_SystemAPI);
		v7_array_push(v7, args, v7_mk_boolean(v7, isFinal?1:0));		
		if(v7_apply(v7, func_Init, scriptRCOMFactory, args, &result)==V7_OK) // call Init
		{
			if(v7_is_boolean(result))
			{
				return v7_get_bool(v7,result)==0?false:true;
			}else
			{
				scriptError=true;
				AddToLog("invalid value returned by \"Init\" method of Factory object");				
			}
		}else
		{
			scriptError=true;
			AddToLog(v7_get_parser_error(v7),"execute error, \"Init\" method of Factory object");
		}		

		return false;
	}

	void RCALL Exec()
	{
		if(scriptError)
			return;

		v7_val_t result, args;
		args = v7_mk_array(v7); // empty params
		if(v7_apply(v7, func_Exec, scriptRCOMFactory, args, &result)!=V7_OK) // call Exec
		{
			scriptError=true;
			AddToLog(v7_get_parser_error(v7), "execute error, \"Exec\" method of Factory object");
		}
	}

	RCOMObject* RCALL QueryObject(const GUID *clsID)
	{
		if(scriptError)		
			return 0;	

		char *strGUID = 0;
		UuidToStringA((UUID*)clsID, &strGUID);
		v7_val_t jsGUID = v7_mk_string(v7, (const char *)strGUID, ~0, 1);
		RpcStringFreeA(&strGUID);

		v7_val_t result, args;
		args = v7_mk_array(v7); // params
		v7_array_push(v7, args, jsGUID);

		if (v7_apply(v7, func_QueryObject, scriptRCOMFactory, args, &result) == V7_OK) // call
		{
			if (v7_is_object(result))
			{
				RCOMObject *obj = (RCOMObject*)v7_get_user_data(v7, result);
				if(obj)
				{
					return obj;
				}
			}
			else if (v7_is_null(result))
			{
				return 0;
			}

			AddToLog("invalid value returned from \"QueryObject\" method of Factory object");
		}
		else
		{
			AddToLog(v7_get_parser_error(v7), "execute error, \"QueryObject\" method of Factory object");
		}
		return 0;
	}

	void RCALL Stop()
	{
		if(scriptError)
			return;

		v7_val_t result, args;
		args = v7_mk_array(v7); // empty params
		if(v7_apply(v7, func_Stop, scriptRCOMFactory, args, &result)!=V7_OK) // call Stop
		{
			scriptError=true;
			AddToLog(v7_get_parser_error(v7), "execute error, \"Stop\" method of Factory object");
		}		
	}

	void RCALL Quit()
	{
		if(scriptError)
			return;	

		v7_val_t result, args;
		args = v7_mk_array(v7); // empty params
		if(v7_apply(v7, func_Quit, scriptRCOMFactory, args, &result)!=V7_OK) // call Quit
		{
			scriptError=true;
			AddToLog(v7_get_parser_error(v7), "execute error, \"Quit\" method of Factory object");
		}				
	}

	const GUID* RCALL GetFactoryID()
	{
		if(scriptError)
			return &CLSID_JSFactory_Default;
		return &factoryID;
	}


};

v7_val_t GenerateJSParameter(struct v7 *v7, int paramType, void **params, int paramIndex)
{
	if(paramType==INT_TYPE)
	{
		return v7_mk_number(v7,*(int*)params[paramIndex]);
	}else if(paramType==FLOAT_TYPE)
	{
		return v7_mk_number(v7,*(float*)params[paramIndex]);
	}else if(paramType==DOUBLE_TYPE)
	{
		return v7_mk_number(v7,*(double*)params[paramIndex]);
	}else if(paramType==BOOL_TYPE)
	{
		return v7_mk_boolean(v7,*(bool*)params[paramIndex]?1:0);
	}else if(paramType==REFERENCE_TYPE)
	{
		if((*(void**)params[paramIndex])==NULL)
			return v7_mk_null();
		return v7_mk_number(v7,(int)*(void**)params[paramIndex]);
	}else if(paramType==STRING_TYPE)
	{
		if((*(RCOMString**)params[paramIndex])==NULL)
			return v7_mk_null();

		const char* text=(*(RCOMString**)params[paramIndex])->GetString();
		v7_val_t retVal=v7_mk_string(v7,text,~0,1);

		return retVal;
	}else if(paramType==RCOM_OBJECT_TYPE)
	{
		if((*(RCOMObject**)params[paramIndex])==NULL)
			return v7_mk_null();

		return ConvertRCOMObjectToJS(v7, *(RCOMObject**)params[paramIndex]);
	}

	return v7_mk_null();
}


class JSRCOMObject : public RCOMObject
{
protected:
  	DEFINE_RCOM_OBJECT_PROPERTIES 
  	
  	virtual ~JSRCOMObject()
  	{
		v7_disown(v7, &jsObject);

  		RELEASE_RCOM_OBJECT_PROPERTIES	
  	}

  	JSFactory *factory;

  	struct v7 *v7;
  	v7_val_t jsObject;
  	v7_val_t func_GetInterfaceIndex;
  	v7_val_t func__GetTypeInfo;
  	v7_val_t func__Invoke;

  	GUID classID;
 public:

 	JSRCOMObject(struct v7 *v7, v7_val_t jsObject, v7_val_t func_GetInterfaceIndex, v7_val_t func__GetTypeInfo, v7_val_t func__Invoke, GUID classID)
 	{
 		INITIALIZE_RCOM_OBJECT_PROPERTIES

 		this->v7=v7;
 		this->jsObject=jsObject;
 		this->func_GetInterfaceIndex=func_GetInterfaceIndex;
 		this->func__GetTypeInfo=func__GetTypeInfo;
 		this->func__Invoke=func__Invoke;
 		this->classID=classID;

		v7_own(v7, &this->jsObject);

 		v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  		factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);
 	}

 	int RCALL _Invoke(int interfaceIndex, int methodID, void **params, int nparam, void *retval)
 	{
 		v7_val_t jsParams=v7_mk_array(v7);
		v7_own(v7, &jsParams);

 		bool failed=false;
 		for(int i=0;i<nparam;i++)
 		{
 			int type=_GetTypeInfo(interfaceIndex,methodID,i);
 			if( (type==UNSUPPORTED_INTERFACE) || (type==UNSUPPORTED_METHOD) || (type==INVALID_PARAM) )
 			{
 				failed=true;
 				break;
 			}

 			v7_array_push(v7, jsParams, GenerateJSParameter(v7,type,params,i));
 		}

		if (failed)
		{
			v7_disown(v7, &jsParams);
			return UNSUPPORTED_INTERFACE;
		}
 			

 		v7_val_t msgObject=v7_mk_object(v7);
 		v7_set(v7, msgObject, "interfaceIndex", 14, v7_mk_number(v7,interfaceIndex));
 		v7_set(v7, msgObject, "methodID", 8, v7_mk_number(v7,methodID));
 		v7_set(v7, msgObject, "params", 6, jsParams);
 		v7_set(v7, msgObject, "retVal", 6, v7_mk_null());

		v7_disown(v7, &jsParams);
		v7_own(v7, &msgObject);

  		v7_val_t result, args;
		args = v7_mk_array(v7); // function params
		v7_array_push(v7, args, msgObject);		

		if(v7_apply(v7, func__Invoke, jsObject, args, &result)==V7_OK) // call
		{
			if(v7_is_number(result)==0)
			{
				factory->AddToLog("invalid value returned by \"_Invoke\" method of JSRCOM object");
				v7_disown(v7, &msgObject);
				return UNSUPPORTED_INTERFACE;
			}else if( v7_get_int(v7,result) != DISPATCH_SUCCESS) // dispatch failed
			{
				v7_disown(v7, &msgObject);
				return v7_get_int(v7,result);
			}

			// cast returned value into c value
			int retType=_GetTypeInfo(interfaceIndex,methodID,RETURN_TYPE_INDEX);
			v7_val_t jsRetVal=v7_get(v7,msgObject,"retVal",6);

			if(retType==INT_TYPE)
			{
				if(v7_is_number(jsRetVal))
				{
					*((int*)retval) = v7_get_int(v7,jsRetVal);
					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;
				}	
			}else if(retType==FLOAT_TYPE)
			{
				if(v7_is_number(jsRetVal))
				{
					*((float*)retval) = (float)v7_get_double(v7,jsRetVal);
					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;
				}	
			}else if(retType==DOUBLE_TYPE)
			{
				if(v7_is_number(jsRetVal))
				{
					*((double*)retval) = v7_get_double(v7,jsRetVal);
					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;
				}	
			}else if(retType==BOOL_TYPE)
			{
				if(v7_is_boolean(jsRetVal))
				{
					*((bool*)retval) = v7_get_bool(v7,jsRetVal)==0?false:true;
					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;
				}	
			}else if(retType==VOID_TYPE)
			{
				v7_disown(v7, &msgObject);
				return DISPATCH_SUCCESS;	
			}else if(retType==REFERENCE_TYPE)
			{
				if(v7_is_null(jsRetVal))
				{
					*((void**)retval) = NULL;
					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;	
				}else if(v7_is_number(jsRetVal))
				{
					*((void**)retval) = (void*)v7_get_int(v7,jsRetVal); // this causes "32bit only code" (coz int is 32bit)
					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;					
				}
			}else if(retType==STRING_TYPE)
			{
				if(v7_is_null(jsRetVal))
				{
					*(RCOMString**)retval = NULL;
					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;	
				}else if(v7_is_string(jsRetVal))
				{
					const char *text=v7_get_cstring(v7, &jsRetVal);
					*(RCOMString**)retval = MakeString(text);

					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;	
				}
			}else if(retType==RCOM_OBJECT_TYPE)
			{
				if(v7_is_null(jsRetVal))
				{
					*(RCOMObject**)retval = NULL;
					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;	
				}else if(v7_is_object(jsRetVal))
				{
					*(RCOMObject**)retval = (RCOMObject*)v7_get_user_data(v7, jsRetVal);
					v7_disown(v7, &msgObject);
					return DISPATCH_SUCCESS;
				}
			}

			factory->AddToLog("invalid value assigned to retVal variable by \"_Invoke\" method of JSRCOM object");
		}else
		{
			factory->AddToLog(v7_get_parser_error(v7), "execute error, \"_Invoke\" method of JSRCOM object");
		}

		v7_disown(v7, &msgObject);
		return UNSUPPORTED_INTERFACE;
 	}

 	int RCALL _GetTypeInfo(int interfaceIndex, int methodID, int paramIndex)
 	{
 		v7_val_t result, args;
		args = v7_mk_array(v7); // params
		v7_array_push(v7, args, v7_mk_number(v7,interfaceIndex));
		v7_array_push(v7, args, v7_mk_number(v7,methodID));
		v7_array_push(v7, args, v7_mk_number(v7,paramIndex));	

  	 	if(v7_apply(v7, func__GetTypeInfo, jsObject, args, &result)==V7_OK) // call
  	 	{
  	 		if(v7_is_number(result))
  	 		{
  	 			return v7_get_int(v7,result);
  	 		}else
  	 		{
  	 			factory->AddToLog("invalid value returned from \"_GetTypeInfo\" method of JSRCOM object");
  	 		}
  	 	}else
  	 	{
			factory->AddToLog(v7_get_parser_error(v7), "execute error, \"_GetTypeInfo\" method of JSRCOM object");
  	 	}

  	 	return UNSUPPORTED_INTERFACE;	
 	}

 	int RCALL GetInterfaceIndex(const GUID *interfaceID)
 	{
 		int retVal=UNSUPPORTED_INTERFACE;

 		char *strGUID=0;
		UuidToStringA((UUID*)interfaceID,&strGUID);
		v7_val_t jsGUID=v7_mk_string(v7, (const char *)strGUID, ~0, 1);

		v7_val_t result, args;
		args = v7_mk_array(v7); // params
		v7_array_push(v7, args, jsGUID);
  	 	
  	 	if(v7_apply(v7, func_GetInterfaceIndex, jsObject, args, &result)==V7_OK) // call
  	 	{
  	 		if(v7_is_number(result))
  	 		{
  	 			retVal=v7_get_int(v7,result);
  	 		}else
  	 		{
  	 			factory->AddToLog("invalid value returned from \"GetInterfaceIndex\" method of JSRCOM object");
  	 		}
  	 	}else
  	 	{
			factory->AddToLog(v7_get_parser_error(v7), "execute error, \"GetInterfaceIndex\" method of JSRCOM object");
  	 	}

  		RpcStringFreeA(&strGUID);
  		return retVal;
 	}

 	const GUID* RCALL GetClassID()
 	{
 		return &classID;
 	}

 	long RCALL AddRef() 
	{ 
		ADD_REF_CODE 
	} 

	long RCALL Release() 
	{ 
		RELEASE_CODE 
	}

	void RCALL OverrideInvoke(RCOMObject *newObject) 
	{ 
		if(!overrideObj) 
		{ 
			newObject->SetParent((RCOMObject*)this); 
		}else 
		{ 
			overrideObj->OverrideInvoke(newObject); 
			overrideObj->Release(); 
		}
		newObject->AddRef(); 
		overrideObj=newObject; 
	}

	bool RCALL SetParent(RCOMObject *parent) 
	{ 
		if(this->parent) 
			return false; 
 		
		parent->AddRef(); 
		this->parent=parent; 

		v7_val_t jsRCOMObject=ConvertRCOMObjectToJS(v7, parent);
		v7_set(v7, jsObject, "parent", 6, jsRCOMObject);
		
		return true; 
	}

	int RCALL GetTypeInfo(int interfaceIndex, int methodID, int paramIndex) 
	{ 
		if(!overrideObj) 
			return _GetTypeInfo(interfaceIndex,methodID,paramIndex); 
		return overrideObj->GetTypeInfo(interfaceIndex,methodID,paramIndex); 
	}

	int RCALL Invoke(int interfaceIndex, int methodID, void **params, int nparam, void *retval) 
	{ 
		if(!overrideObj) 
			return _Invoke(interfaceIndex,methodID,params,nparam,retval); 
		return overrideObj->Invoke(interfaceIndex,methodID,params,nparam,retval); 
	}
};

static enum v7_err js_Alert(struct v7 *v7, v7_val_t *res) 
{
	v7_val_t param1=v7_arg(v7, 0);
	v7_own(v7, &param1); // just for sure

	if(v7_is_string(param1))
  	{
  		printf("[Alert]: %s\n",v7_get_cstring(v7, &param1));
  	}else
  	{
  		v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  		JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

  		factory->AddToLog("invalid parameter passed for \"alert\" function");
  	}

	v7_disown(v7, &param1);
  	*res = v7_mk_number(v7, 0);
	return V7_OK;
}

static enum v7_err js_Print(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1 = v7_arg(v7, 0);
	v7_own(v7, &param1); // just for sure

	v7_val_t cobj_Factory = v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
	JSFactory *factory = (JSFactory*)v7_get_user_data(v7, cobj_Factory);

	if (v7_is_string(param1))
	{
		factory->AddToLog(v7_get_cstring(v7, &param1),"print");
	}
	else
	{
		factory->AddToLog("invalid parameter passed for \"print\" function");
	}

	v7_disown(v7, &param1);
	*res = v7_mk_number(v7, 0);
	return V7_OK;
}

static enum v7_err js_Inlcude(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1 = v7_arg(v7, 0);
	v7_own(v7, &param1); // just for sure

	if (v7_is_string(param1))
	{
		v7_val_t exec_result;
		enum v7_err err = v7_exec_file(v7, v7_get_cstring(v7, &param1), &exec_result);

		if (err != V7_OK)
		{
			v7_val_t cobj_Factory = v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
			JSFactory *factory = (JSFactory*)v7_get_user_data(v7, cobj_Factory);
			factory->AddToLog(v7_get_parser_error(v7), v7_get_cstring(v7, &param1));
		}
	}
	else
	{
		v7_val_t cobj_Factory = v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
		JSFactory *factory = (JSFactory*)v7_get_user_data(v7, cobj_Factory);
		factory->AddToLog("invalid parameter passed for \"inlcude\" function");
	}

	v7_disown(v7, &param1);
	*res = v7_mk_number(v7, 0);
	return V7_OK;
}

static enum v7_err js_ToStringArray(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1 = v7_arg(v7, 0);
	v7_val_t param2 = v7_arg(v7, 1);

	if (v7_is_number(param1))
	{
		if (v7_is_number(param2))
		{
			RCOMString **strList = (RCOMString**)v7_get_int(v7, param1);
			int len = v7_get_int(v7, param2);

			v7_val_t jsList = v7_mk_array(v7);		

			for (int i = 0; i < len; i++)
			{
				RCOMString *str = strList[i];
				if (str)
				{
					const char* text = str->GetString();
					v7_val_t jsStr = v7_mk_string(v7, text, ~0, 1);
					v7_array_push(v7, jsList, jsStr);
				}else
				{
					v7_array_push(v7, jsList, v7_mk_null());
				}
			}

			*res = jsList;
			return V7_OK;
		}
	}

	v7_val_t cobj_Factory = v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
	JSFactory *factory = (JSFactory*)v7_get_user_data(v7, cobj_Factory);

	factory->AddToLog("invalid parameter passed for \"toStringArray\" function");

	*res = v7_mk_null();
	return V7_OK;
}

static enum v7_err js_ToIntArray(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1 = v7_arg(v7, 0);
	v7_val_t param2 = v7_arg(v7, 1);

	if (v7_is_number(param1))
	{
		if (v7_is_number(param2))
		{
			int *intList = (int*)v7_get_int(v7, param1);
			int len = v7_get_int(v7, param2);

			v7_val_t jsList = v7_mk_array(v7);

			for (int i = 0; i < len; i++)
			{
				int val = intList[i];
				v7_array_push(v7, jsList, v7_mk_number(v7, val));
			}

			*res = jsList;
			return V7_OK;
		}
	}

	v7_val_t cobj_Factory = v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
	JSFactory *factory = (JSFactory*)v7_get_user_data(v7, cobj_Factory);

	factory->AddToLog("invalid parameter passed for \"toIntArray\" function");

	*res = v7_mk_null();
	return V7_OK;
}

static enum v7_err js_ToFloatArray(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1 = v7_arg(v7, 0);
	v7_val_t param2 = v7_arg(v7, 1);

	if (v7_is_number(param1))
	{
		if (v7_is_number(param2))
		{
			float *floatList = (float*)v7_get_int(v7, param1);
			int len = v7_get_int(v7, param2);

			v7_val_t jsList = v7_mk_array(v7);

			for (int i = 0; i < len; i++)
			{
				float val = floatList[i];
				v7_array_push(v7, jsList, v7_mk_number(v7, val));
			}

			*res = jsList;
			return V7_OK;
		}
	}

	v7_val_t cobj_Factory = v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
	JSFactory *factory = (JSFactory*)v7_get_user_data(v7, cobj_Factory);

	factory->AddToLog("invalid parameter passed for \"toFloatArray\" function");

	*res = v7_mk_null();
	return V7_OK;
}

static enum v7_err js_ToDoubleArray(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1 = v7_arg(v7, 0);
	v7_val_t param2 = v7_arg(v7, 1);

	if (v7_is_number(param1))
	{
		if (v7_is_number(param2))
		{
			double *doubleList = (double*)v7_get_int(v7, param1);
			int len = v7_get_int(v7, param2);

			v7_val_t jsList = v7_mk_array(v7);

			for (int i = 0; i < len; i++)
			{
				double val = doubleList[i];
				v7_array_push(v7, jsList, v7_mk_number(v7, val));
			}

			*res = jsList;
			return V7_OK;
		}
	}

	v7_val_t cobj_Factory = v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
	JSFactory *factory = (JSFactory*)v7_get_user_data(v7, cobj_Factory);

	factory->AddToLog("invalid parameter passed for \"toDoubleArray\" function");

	*res = v7_mk_null();
	return V7_OK;
}

static enum v7_err js_ToBoolArray(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1 = v7_arg(v7, 0);
	v7_val_t param2 = v7_arg(v7, 1);

	if (v7_is_number(param1))
	{
		if (v7_is_number(param2))
		{
			bool *boolList = (bool*)v7_get_int(v7, param1);
			int len = v7_get_int(v7, param2);

			v7_val_t jsList = v7_mk_array(v7);

			for (int i = 0; i < len; i++)
			{
				bool val = boolList[i];
				v7_array_push(v7, jsList, v7_mk_boolean(v7, val?1:0));
			}

			*res = jsList;
			return V7_OK;
		}
	}

	v7_val_t cobj_Factory = v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
	JSFactory *factory = (JSFactory*)v7_get_user_data(v7, cobj_Factory);

	factory->AddToLog("invalid parameter passed for \"toBoolArray\" function");

	*res = v7_mk_null();
	return V7_OK;
}

static enum v7_err js_ToRCOMObjectArray(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1 = v7_arg(v7, 0);
	v7_val_t param2 = v7_arg(v7, 1);

	if (v7_is_number(param1))
	{
		if (v7_is_number(param2))
		{
			RCOMObject **objList = (RCOMObject**)v7_get_int(v7, param1);
			int len = v7_get_int(v7, param2);

			v7_val_t jsList = v7_mk_array(v7);

			for (int i = 0; i < len; i++)
			{
				RCOMObject *obj = objList[i];
				if (obj)
				{
					v7_val_t jsObj=ConvertRCOMObjectToJS(v7, obj);
					v7_array_push(v7, jsList, jsObj);
				}
				else
				{
					v7_array_push(v7, jsList, v7_mk_null());
				}
				
			}

			*res = jsList;
			return V7_OK;
		}
	}

	v7_val_t cobj_Factory = v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
	JSFactory *factory = (JSFactory*)v7_get_user_data(v7, cobj_Factory);

	factory->AddToLog("invalid parameter passed for \"toRCOMObjectArray\" function");

	*res = v7_mk_null();
	return V7_OK;
}

// we have to create js side & rcom side
static enum v7_err js_JSObjectToRCOMObject(struct v7 *v7, v7_val_t *res) 
{
	v7_val_t jsObject=v7_arg(v7, 0);
	v7_own(v7, &jsObject);

	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

	if(v7_is_object(jsObject))
	{
		// check for essential 4 methods
		v7_val_t func_GetInterfaceIndex = v7_get(v7, jsObject, "GetInterfaceIndex", 17); // query for GetInterfaceIndex method in that obj
		if(func_GetInterfaceIndex!=V7_UNDEFINED)
		{
			v7_val_t func__GetTypeInfo = v7_get(v7, jsObject, "_GetTypeInfo", 12); // query for _GetTypeInfo method in that obj
			if(func__GetTypeInfo!=V7_UNDEFINED)
			{
				v7_val_t func__Invoke = v7_get(v7, jsObject, "_Invoke", 7); // query for _Invoke method in that obj
				if(func__Invoke!=V7_UNDEFINED)
				{
					v7_val_t func_GetClassID = v7_get(v7, jsObject, "GetClassID", 10); // query for GetClassID method in that obj
					if(func_GetClassID!=V7_UNDEFINED)
					{
						v7_val_t args = v7_mk_array(v7);
						v7_val_t result;
						if(v7_apply(v7, func_GetClassID, jsObject, args, &result)==V7_OK) // call GetClassID
			 			{
			 				const char *strGUID=v7_get_cstring(v7, &result);
			 				if(strGUID!=NULL)
			 				{
			 					GUID classID;
			 					// convert guid string to guid
			 					if(UuidFromStringA((char*)strGUID,&classID)==RPC_S_OK)
			 					{
									v7_disown(v7, &jsObject);

			 						JSRCOMObject *jsRCOMObject=new JSRCOMObject(v7,jsObject,func_GetInterfaceIndex,func__GetTypeInfo,func__Invoke,classID);
			 						v7_set_user_data(v7, jsObject, (RCOMObject*)jsRCOMObject);

			 						// setup parent property of js object
			 						v7_set(v7, jsObject, "parent", 6, v7_mk_null());
			 						
			 						// add remaining js methods
									v7_set_method(v7, jsObject, "AddRef", &rcom_AddRef);
									v7_set_method(v7, jsObject, "Release", &rcom_Release);	
									v7_set_method(v7, jsObject, "GetTypeInfo", &rcom_GetTypeInfo);
									v7_set_method(v7, jsObject, "Invoke", &rcom_Invoke);
									v7_set_method(v7, jsObject, "OverrideInvoke", &rcom_OverrideInvoke);
									v7_set_method(v7, jsObject, "SetParent", &rcom_SetParent);
			 						
			 						*res = v7_mk_boolean(v7, 1);
									return V7_OK;  	
			 					}
			 				}
			 				factory->AddToLog("convertToRCOMObject: \"GetClassID\" method returned invalid guid");
			 			}else
						{
							factory->AddToLog("convertToRCOMObject: error occured while executing \"GetClassID\" method");
						}
					}else
					{
						factory->AddToLog("convertToRCOMObject: \"GetClassID\" method is not defined");
					}
				}else
				{
					factory->AddToLog("convertToRCOMObject: \"_Invoke\" method is not defined");
				}
			}else
			{
				factory->AddToLog("convertToRCOMObject: \"_GetTypeInfo\" method is not defined");
			}
		}else
		{
			factory->AddToLog("convertToRCOMObject: \"GetInterfaceIndex\" method is not defined");
		}
	}else
	{
		factory->AddToLog("invalid parameter passed for \"convertToRCOMObject\" function");
	}

	v7_disown(v7, &jsObject);
  	*res = v7_mk_boolean(v7, 0);
	return V7_OK;  	
}

static enum v7_err api_ShutdownSystem(struct v7 *v7, v7_val_t *res)
{
	bool retVal=systemAPI->ShutdownSystem();
  	*res = v7_mk_boolean(v7, retVal?1:0);
	return V7_OK;	
}

static enum v7_err api_IsFactoryAvailable(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1=v7_arg(v7, 0);

	if(v7_is_string(param1))
  	{
  		const char *strGUID=v7_get_cstring(v7, &param1);
  		if(strGUID!=NULL)
		{
			GUID factoryID;
			// convert guid string to guid
			if(UuidFromStringA((char*)strGUID,&factoryID)==RPC_S_OK)
			{
				bool retVal=systemAPI->IsFactoryAvailable(&factoryID);
			  	*res = v7_mk_boolean(v7, retVal?1:0);
				return V7_OK;				
			}
		}
  	}

  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

  	factory->AddToLog("invalid parameter passed for \"IsFactoryAvailable\" API");

  	*res = v7_mk_boolean(v7, 0);
	return V7_OK;  	
}

static enum v7_err rcom_AddRef(struct v7 *v7, v7_val_t *res)
{
	RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));
  	*res = v7_mk_number(v7, object->AddRef());
	return V7_OK; 	
}

static enum v7_err rcom_Release(struct v7 *v7, v7_val_t *res)
{
	RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));
  	*res = v7_mk_number(v7, object->Release());
	return V7_OK; 	
}

static enum v7_err rcom_GetClassID(struct v7 *v7, v7_val_t *res)
{
	RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));
	const GUID *clsID=object->GetClassID();
	char *strGUID=0;
	UuidToStringA((UUID*)clsID,&strGUID);
  	*res =  v7_mk_string(v7, (const char *)strGUID, ~0, 1);
  	RpcStringFreeA(&strGUID);
	return V7_OK; 	
}

static enum v7_err rcom_GetInterfaceIndex(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1=v7_arg(v7, 0);

	if(v7_is_string(param1))
  	{
  		const char *strGUID=v7_get_cstring(v7, &param1);
  		if(strGUID!=NULL)
		{
			GUID interfaceID;
			// convert guid string to guid
			if(UuidFromStringA((char*)strGUID,&interfaceID)==RPC_S_OK)
			{
				RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));
				int retVal=object->GetInterfaceIndex(&interfaceID);
			  	*res = v7_mk_number(v7, retVal);
				return V7_OK;				
			}
		}
  	}

  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

  	factory->AddToLog("invalid parameter passed for \"GetInterfaceIndex\" method");

  	*res = v7_mk_number(v7, UNSUPPORTED_INTERFACE);
	return V7_OK; 	
}

static enum v7_err rcom_GetTypeInfo(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1=v7_arg(v7, 0);
	v7_val_t param2=v7_arg(v7, 1);
	v7_val_t param3=v7_arg(v7, 2);	

	if(v7_is_number(param1))
	{
		if(v7_is_number(param2))
		{
			if(v7_is_number(param3))
			{
				RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));
				int retVal=object->GetTypeInfo(v7_get_int(v7,param1),v7_get_int(v7,param2),v7_get_int(v7,param3));
			  	*res = v7_mk_number(v7, retVal);
				return V7_OK;	
			}
		}
	}

  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);
 
  	factory->AddToLog("invalid parameter passed for \"GetTypeInfo\" method");

  	*res = v7_mk_number(v7, UNSUPPORTED_INTERFACE);
	return V7_OK; 

}

static enum v7_err rcom__GetTypeInfo(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1=v7_arg(v7, 0);
	v7_val_t param2=v7_arg(v7, 1);
	v7_val_t param3=v7_arg(v7, 2);	

	if(v7_is_number(param1))
	{
		if(v7_is_number(param2))
		{
			if(v7_is_number(param3))
			{
				RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));
				int retVal=object->_GetTypeInfo(v7_get_int(v7,param1),v7_get_int(v7,param2),v7_get_int(v7,param3));
			  	*res = v7_mk_number(v7, retVal);
				return V7_OK;	
			}
		}
	}

  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

  	factory->AddToLog("invalid parameter passed for \"_GetTypeInfo\" method");

  	*res = v7_mk_number(v7, UNSUPPORTED_INTERFACE);
	return V7_OK; 

}

void* GenerateCParameter(struct v7 *v7,int paramType,int paramIndex,v7_val_t js_params)
{
	if(paramType==INT_TYPE)
	{
		v7_val_t val=v7_array_get(v7,js_params,paramIndex);
		if(v7_is_number(val))
		{
			int *param=(int*)malloc(sizeof(int));
			*param=v7_get_int(v7,val);
			return param;
		}
	}else if(paramType==STRING_TYPE)
	{
		v7_val_t val=v7_array_get(v7,js_params,paramIndex);
		if(v7_is_null(val))
		{
			RCOMString **param=(RCOMString**)malloc(sizeof(RCOMString*));
			*param=NULL;

			return param;			
		}else if(v7_is_string(val))
		{
			const char *cstr=v7_get_cstring(v7,&val);

			RCOMString **param=(RCOMString**)malloc(sizeof(RCOMString*));
			*param=MakeString(cstr);

			return param;
		}
	}else if(paramType==FLOAT_TYPE)
	{
		v7_val_t val=v7_array_get(v7,js_params,paramIndex);
		if(v7_is_number(val))
		{
			float *param=(float*)malloc(sizeof(float));
			*param=(float)v7_get_double(v7,val);
			return param;
		}		
	}else if(paramType==DOUBLE_TYPE)
	{
		v7_val_t val=v7_array_get(v7,js_params,paramIndex);
		if(v7_is_number(val))
		{
			double *param=(double*)malloc(sizeof(double));
			*param=(double)v7_get_double(v7,val);
			return param;
		}		
	}else if(paramType==BOOL_TYPE)
	{
		v7_val_t val=v7_array_get(v7,js_params,paramIndex);
		if(v7_is_boolean((val)))
		{
			bool *param=(bool*)malloc(sizeof(bool));
			*param=v7_get_bool(v7,val)==0?false:true;
			return param;
		}		
	}else if(paramType==REFERENCE_TYPE)
	{
		v7_val_t val=v7_array_get(v7,js_params,paramIndex);
		if(v7_is_null(val))
		{
			void **param=(void**)malloc(sizeof(void*));
			*param=NULL;
			return param;
		}else if(v7_is_number(val))
		{
			void **param=(void**)malloc(sizeof(void*));
			*param=(void*)v7_get_int(v7,val);
			return param;
		}
	}else if(paramType==RCOM_OBJECT_TYPE)
	{
		v7_val_t val=v7_array_get(v7,js_params,paramIndex);
		if(v7_is_null(val))
		{
			RCOMObject **param=(RCOMObject**)malloc(sizeof(void*));
			*param=NULL;
			return param;
		}else if(v7_is_object(val))
		{
			RCOMObject **param=(RCOMObject**)malloc(sizeof(void*));
			*param=(RCOMObject*)v7_get_user_data(v7, val);
			return param;
		}		
	}

	return 0;
}

void* AllocateReturnType(int returnType)
{
	if(returnType==INT_TYPE)
	{
		return malloc(sizeof(int));
	}else if(returnType==FLOAT_TYPE)
	{
		return malloc(sizeof(float));
	}else if(returnType==DOUBLE_TYPE)
	{
		return malloc(sizeof(double));
	}else if(returnType==BOOL_TYPE)
	{
		return malloc(sizeof(bool));
	}else if(returnType==REFERENCE_TYPE)
	{
		return malloc(sizeof(void*));
	}else if(returnType==STRING_TYPE)
	{
		return malloc(sizeof(RCOMString*));
	}else if(returnType==RCOM_OBJECT_TYPE)
	{
		return malloc(sizeof(RCOMObject*));
	}
	return 0;
}

void SetReturnValueForJS(struct v7 *v7,v7_val_t msgObject, int retType, void *retVal)
{
	if(retType==INT_TYPE)
	{
		v7_set(v7,msgObject,"retVal",6,v7_mk_number(v7,*(int*)retVal));
	}else if(retType==FLOAT_TYPE)
	{
		v7_set(v7,msgObject,"retVal",6,v7_mk_number(v7,*(float*)retVal));
	}else if(retType==DOUBLE_TYPE)
	{
		v7_set(v7,msgObject,"retVal",6,v7_mk_number(v7,*(double*)retVal));
	}else if(retType==BOOL_TYPE)
	{
		v7_set(v7,msgObject,"retVal",6,v7_mk_number(v7,*(bool*)retVal));
	}else if(retType==REFERENCE_TYPE)
	{
		if(*(void**)retVal==NULL)
			v7_set(v7,msgObject,"retVal",6,v7_mk_null());
		else
			v7_set(v7,msgObject,"retVal",6,v7_mk_number(v7,*(int*)retVal));
	}else if(retType==STRING_TYPE)
	{
		if(*(RCOMString**)retVal==NULL)
		{
			v7_set(v7,msgObject,"retVal",6,v7_mk_null());
		}
		else
		{
			const char *aText=(*(RCOMString**)retVal)->GetString();
			v7_set(v7,msgObject,"retVal",6,v7_mk_string(v7,aText,~0,1));

			(*(RCOMString**)retVal)->Release();
		}
	}else if(retType==RCOM_OBJECT_TYPE)
	{
		if(*(RCOMObject**)retVal==NULL)
			v7_set(v7,msgObject,"retVal",6,v7_mk_null());
		else
			v7_set(v7,msgObject,"retVal",6,ConvertRCOMObjectToJS(v7, *(RCOMObject**)retVal));
	}
}

static enum v7_err rcom_Invoke(struct v7 *v7, v7_val_t *res)
{
	v7_val_t msgObject=v7_arg(v7, 0);
	v7_own(v7, &msgObject);

	if(v7_is_object(msgObject))
	{
		v7_val_t js_interfaceIndex=v7_get(v7, msgObject, "interfaceIndex",14);
		if(v7_is_number(js_interfaceIndex))
		{
			int interfaceIndex=v7_get_int(v7,js_interfaceIndex);

			v7_val_t js_methodID=v7_get(v7, msgObject, "methodID",8);
			if(v7_is_number(js_methodID))
			{
				int methodID=v7_get_int(v7,js_methodID);

				v7_val_t js_params=v7_get(v7, msgObject, "params",6);
				if(v7_is_array(v7,js_params))
				{
					int nparam=v7_array_length(v7,js_params);

					if(nparam<11) // js for rcom is limited to 10 params
					{
						RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));

						bool errorOccured=false;
						void* params[10];
						for(int i=0;i<nparam;i++)
						{
							int paramType=object->GetTypeInfo(interfaceIndex,methodID,i);
							params[i]=GenerateCParameter(v7,paramType,i,js_params);	
							if(params[i]==0)
							{
								errorOccured=true;
								break;
							}
						}

						if(!errorOccured)
						{
							int retType=object->GetTypeInfo(interfaceIndex,methodID,RETURN_TYPE_INDEX);
							void *retVal=AllocateReturnType(retType);
							int result=object->Invoke(interfaceIndex,methodID,params,nparam,retVal);

							SetReturnValueForJS(v7,msgObject,retType,retVal);

							// free retVal value
							if(retVal)
								free(retVal);

							*res = v7_mk_number(v7, result);
						}

						// free allocated params
						for(int i=0;i<nparam;i++)
						{
							if(params[i]==0)
								break;

							int paramType=object->GetTypeInfo(interfaceIndex,methodID,i);
							if(paramType==STRING_TYPE)
							{
								if( (*((RCOMString**)params[i]))!=NULL )
									(*((RCOMString**)params[i]))->Release();
							}

							free(params[i]);
						}

						if (!errorOccured)
						{
							v7_disown(v7, &msgObject);
							return V7_OK;
						}

					}
				}
			}
		}
	}

  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

  	factory->AddToLog("invalid parameter passed for \"Invoke\" method");

	v7_disown(v7, &msgObject);
  	*res = v7_mk_number(v7, UNSUPPORTED_INTERFACE);
	return V7_OK; 	
}

static enum v7_err rcom__Invoke(struct v7 *v7, v7_val_t *res)
{
	v7_val_t msgObject=v7_arg(v7, 0);
	v7_own(v7, &msgObject);

	if(v7_is_object(msgObject))
	{
		v7_val_t js_interfaceIndex=v7_get(v7, msgObject, "interfaceIndex",14);
		if(v7_is_number(js_interfaceIndex))
		{
			int interfaceIndex=v7_get_int(v7,js_interfaceIndex);

			v7_val_t js_methodID=v7_get(v7, msgObject, "methodID",8);
			if(v7_is_number(js_methodID))
			{
				int methodID=v7_get_int(v7,js_methodID);

				v7_val_t js_params=v7_get(v7, msgObject, "params",6);
				if(v7_is_array(v7,js_params))
				{
					int nparam=v7_array_length(v7,js_params);

					if(nparam<11) // js for rcom is limited to 10 params
					{
						RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));

						bool errorOccured=false;
						void* params[10];
						for(int i=0;i<nparam;i++)
						{
							int paramType=object->_GetTypeInfo(interfaceIndex,methodID,i);
							params[i]=GenerateCParameter(v7,paramType,i,js_params);	
							if(params[i]==0)
							{
								errorOccured=true;
								break;
							}
						}

						if(!errorOccured)
						{
							int retType=object->_GetTypeInfo(interfaceIndex,methodID,RETURN_TYPE_INDEX);
							void *retVal=AllocateReturnType(retType);
							int result=object->_Invoke(interfaceIndex,methodID,params,nparam,retVal);

							SetReturnValueForJS(v7,msgObject,retType,retVal);

							// free retVal value
							if(retVal)
								free(retVal);

							*res = v7_mk_number(v7, result);
						}

						// free allocated params
						for(int i=0;i<nparam;i++)
						{
							if(params[i]==0)
								break;

							int paramType=object->_GetTypeInfo(interfaceIndex,methodID,i);
							if(paramType==STRING_TYPE)
							{
								if( (*((RCOMString**)params[i]))!=NULL )
									(*((RCOMString**)params[i]))->Release();
							}

							free(params[i]);
						}

						if (!errorOccured)
						{
							v7_disown(v7, &msgObject);
							return V7_OK;
						}

					}
				}
			}
		}
	}

  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

  	factory->AddToLog("invalid parameter passed for \"_Invoke\" method");

	v7_disown(v7, &msgObject);
  	*res = v7_mk_number(v7, UNSUPPORTED_INTERFACE);
	return V7_OK; 	
}

static enum v7_err rcom_OverrideInvoke(struct v7 *v7, v7_val_t *res)
{
	v7_val_t rcomJSObject=v7_arg(v7, 0);
	if(v7_is_object(rcomJSObject))
	{
		RCOMObject *newObject=(RCOMObject*)v7_get_user_data(v7, rcomJSObject);
		RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));
		object->OverrideInvoke(newObject);
		return V7_OK; 
	}	

  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

  	factory->AddToLog("invalid parameter passed for \"OverrideInvoke\" method");
	return V7_OK; 
}

static enum v7_err rcom_SetParent(struct v7 *v7, v7_val_t *res)
{
	v7_val_t rcomJSObject=v7_arg(v7, 0);
	if(v7_is_object(rcomJSObject))
	{
		RCOMObject *parent=(RCOMObject*)v7_get_user_data(v7, rcomJSObject);
		RCOMObject *object=(RCOMObject*)v7_get_user_data(v7, v7_get_this(v7));
		bool retVal=object->SetParent(parent);

		*res = v7_mk_boolean(v7, retVal?1:0);
		return V7_OK; 
	}	

  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

  	factory->AddToLog("invalid parameter passed for \"SetParent\" method");
  	*res = v7_mk_boolean(v7, 0);
	return V7_OK; 
}

// we have to create js side only
v7_val_t ConvertRCOMObjectToJS(struct v7 *v7, RCOMObject *object)
{
	v7_val_t jsObject = v7_mk_object(v7);

	v7_set_user_data(v7, jsObject, object);

	v7_set_method(v7, jsObject, "AddRef", &rcom_AddRef);
	v7_set_method(v7, jsObject, "Release", &rcom_Release);	
	v7_set_method(v7, jsObject, "GetClassID", &rcom_GetClassID);
	v7_set_method(v7, jsObject, "GetInterfaceIndex", &rcom_GetInterfaceIndex);	
	v7_set_method(v7, jsObject, "GetTypeInfo", &rcom_GetTypeInfo);	
	v7_set_method(v7, jsObject, "_GetTypeInfo", &rcom__GetTypeInfo);
	v7_set_method(v7, jsObject, "Invoke", &rcom_Invoke);	
	v7_set_method(v7, jsObject, "_Invoke", &rcom__Invoke);	
	v7_set_method(v7, jsObject, "OverrideInvoke", &rcom_OverrideInvoke);
	v7_set_method(v7, jsObject, "SetParent", &rcom_SetParent);

	return jsObject;
}

static enum v7_err api_GetObjectFromFactory(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1=v7_arg(v7, 0);
	v7_val_t param2=v7_arg(v7, 1);

	if(v7_is_string(param1))
	{
		if(v7_is_string(param2))
		{
  			const char *strFactoryGUID=v7_get_cstring(v7, &param1);		
   			const char *strClassGUID=v7_get_cstring(v7, &param2);
   			if( (strFactoryGUID!=NULL) && (strClassGUID!=NULL) )
   			{
   				GUID factoryID,clsID;
   				if(UuidFromStringA((char*)strFactoryGUID,&factoryID)==RPC_S_OK)
   				{
   		   			if(UuidFromStringA((char*)strClassGUID,&clsID)==RPC_S_OK)
   					{
   						RCOMObject* object = systemAPI->GetObjectFromFactory(&factoryID,&clsID);
   						if(object)
   						{
   							*res = ConvertRCOMObjectToJS(v7, object);
   						}else
   						{
   							*res = v7_mk_null();
   						}
   						return V7_OK;
   					}			
   				}
   			}	
		}	
	}

  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

  	factory->AddToLog("invalid parameter passed for \"GetObjectFromFactory\" API");

  	*res = v7_mk_null();
	return V7_OK;  

}

static enum v7_err api_GetApplicationDir(struct v7 *v7, v7_val_t *res)
{
	const char *aText=systemAPI->GetApplicationDir();
  	*res = v7_mk_string(v7,aText,~0,1);

	return V7_OK;  			
}

static enum v7_err api_GetCommandlineArgCount(struct v7 *v7, v7_val_t *res)
{
  	*res = v7_mk_number(v7, systemAPI->GetCommandlineArgCount());
	return V7_OK; 		
}

static enum v7_err api_GetCommandlineArgument(struct v7 *v7, v7_val_t *res)
{
	v7_val_t param1=v7_arg(v7, 0);
	if(v7_is_number(param1))
	{
		char *arg=systemAPI->GetCommandlineArgument(v7_get_int(v7,param1));
		if(arg)
		{
		  	*res = v7_mk_string(v7,arg,~0,1);
			return V7_OK;  	
		}else
		{
  	  		*res = v7_mk_null();
			return V7_OK; 
		}
	}else
	{
	  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
	  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

	  	factory->AddToLog("invalid parameter passed for \"GetCommandlineArgument\" API");

  	  	*res = v7_mk_null();
		return V7_OK; 		
	}

}

static enum v7_err api_SetNewInstanceListener(struct v7 *v7, v7_val_t *res)
{
	v7_val_t rcomJSObject=v7_arg(v7, 0);
	if(v7_is_object(rcomJSObject))
	{
		RCOMObject *newInstanceListener=(RCOMObject*)v7_get_user_data(v7, rcomJSObject);
		systemAPI->SetNewInstanceListener(newInstanceListener);
		return V7_OK; 	
	}else
	{
	  	v7_val_t cobj_Factory=v7_get(v7, v7_get_global(v7), "cobj_Factory", 12);
	  	JSFactory *factory=(JSFactory*)v7_get_user_data(v7, cobj_Factory);

	  	factory->AddToLog("invalid parameter passed for \"SetNewInstanceListener\" API");

		return V7_OK; 		
	}
}

static enum v7_err api_GetNewInstanceListener(struct v7 *v7, v7_val_t *res)
{
	RCOMObject *instanceListener=systemAPI->GetNewInstanceListener();
	if(instanceListener)
	{
		*res = ConvertRCOMObjectToJS(v7, instanceListener);
		return V7_OK; 
	}else
	{
  	  	*res = v7_mk_null();
		return V7_OK; 			
	}
}

#ifdef WIN32 // windows

void SearchDirForJSFactories(char* root_)
{
	char* buf = new char[MAX_PATH];
	char* root = new char[MAX_PATH];

	GetCurrentDirectoryA(MAX_PATH, buf);
	strcpy(root, root_);

	if (SetCurrentDirectoryA(root) == 0)
	{
		delete[]root;
		delete[]buf;
		return;
	}

	WIN32_FIND_DATAA current_file;
	HANDLE file_search = 0;

	file_search = FindFirstFileA("*.*", &current_file);

	if (file_search == INVALID_HANDLE_VALUE)
	{
		delete[]root;
		delete[]buf;
		return;
	}

	do{
		if (current_file.dwFileAttributes == 0xffffffff)
			continue;

		if ((!lstrcmpA(current_file.cFileName, ".")) || (!lstrcmpA(current_file.cFileName, "..")))
		{
			continue;
		}

		if ((current_file.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0)
		{

			char* root2 = new char[MAX_PATH];
			strcpy(root2, root);

			if (root2[lstrlenA(root2) - 1] != '\\')
			{
				strcat(root2, "\\");
			}
			strcat(root2, current_file.cFileName);

			SearchDirForJSFactories(root2);
		}
		else
		{
			// root is dir & current_file.cFileName is filename			
			char filePath[MAX_PATH];
			strcpy(filePath, root);

			if (filePath[lstrlenA(filePath) - 1] != '\\')
			{
				strcat(filePath, "\\");
			}
			strcat(filePath, current_file.cFileName);

			if (strcmp("main.js", current_file.cFileName) == 0)
			{
				jsCompList.AddPointer(strdup(filePath));
			}
		}
	} while (FindNextFileA(file_search, &current_file));

	FindClose(file_search);

	SetCurrentDirectoryA(buf);
	delete[]buf;
	delete[]root;

}

#else // linux

void SearchDirForJSFactories(char* dir)
{
	// http://www.johnloomis.org/ece537/notes/Files/Examples/printdir.html

    DIR *dp;
    struct dirent *entry;
    struct stat statbuf;

    if((dp = opendir(dir)) == NULL)
        return;

    chdir(dir);
    while((entry = readdir(dp)) != NULL) 
	{
        lstat(entry->d_name,&statbuf);

        if(S_ISDIR(statbuf.st_mode))
		{
            /* Found a directory, but ignore . and .. */
            if( (strcmp(".",entry->d_name) == 0) || (strcmp("..",entry->d_name) == 0) )
                continue;

            /* Recurse at a new indent level */
            SearchDirForJSFactories(entry->d_name);
        }
        else
		{
			if( strcmp(entry->d_name, "main.js") == 0 )
			{
				char filePath[512];
				getcwd(filePath, 512);
				strcat(filePath, "/");
				strcat(filePath, entry->d_name);
				jsCompList.AddPointer(strdup(filePath));
			}
		}
    }

    chdir("..");
    closedir(dp);
}

#endif

RCOM_LIB_EXPORT RCOMFactory* GetRCOMFactory(int index)
{
	if (index == 0)
	{

#ifdef WIN32 // windows

		GetModuleFileNameA(GetModuleHandleA(0), pluginPath, 512); // get executable path

		char *p;
		for (p = pluginPath; *p; p++) {}	// find end
		for (; p > pluginPath && *p != '\\'; p--) {} // back up to last backslash
		*p = 0;	// kill it

		strcat(pluginPath, "\\plugins");

		// js path
		strcpy(jsPluginPath, pluginPath);
		strcat(jsPluginPath, "\\js");

#else // linux

		int size = readlink("/proc/self/exe", pluginPath, 512 - 1);
		pluginPath[size] = 0;

		char *p;
		for (p = pluginPath; *p; p++) { }	// find end
		for (; p > pluginPath && *p != '/'; p--) { } // back up to last backslash
		*p = 0;	// kill it

		strcat(pluginPath, "/plugins");

		// js path
		strcpy(jsPluginPath, pluginPath);
		strcat(jsPluginPath, "/js");

#endif

		SearchDirForJSFactories(jsPluginPath);

		if (jsCompList.GetSize() == 0) // no js factories
			return 0;

		char *jsFactoryPath = jsCompList.GetPointer(0);
		RCOMFactory* factory = new JSFactory(jsFactoryPath);
		return factory;

	}
	else if (index < jsCompList.GetSize())
	{
		char *jsFactoryPath = jsCompList.GetPointer(index);
		RCOMFactory* factory = new JSFactory(jsFactoryPath);
		return factory;
	}

	// delete all jsFactory paths
	for (int i = 0; i < jsCompList.GetSize(); i++)
	{
		free(jsCompList.GetPointer(i));
	}

	return 0;
}

