

/*
	RCOM v1.0 By Hasaranga
 */

#ifndef _RCOM_H_
#define _RCOM_H_

#include<windows.h>

#define UNSUPPORTED_METHOD -4
#define UNSUPPORTED_INTERFACE -3
#define INVALID_PARAM -2	
#define RETURN_TYPE_INDEX -1
#define DISPATCH_SUCCESS 0
#define INT_TYPE 1
#define STRING_TYPE 2
#define FLOAT_TYPE 3
#define DOUBLE_TYPE 4
#define VOID_TYPE 5
#define BOOL_TYPE 6
#define RCOM_OBJECT_TYPE 7
#define REFERENCE_TYPE 8

#define RCALL __stdcall

struct RCOMObject
{
	virtual long RCALL AddRef()=0;
	virtual long RCALL Release()=0;
	virtual const GUID* RCALL GetClassID()=0; // for RTTI
	virtual int RCALL GetInterfaceIndex(const GUID *interfaceID)=0; // returns UNSUPPORTED_INTERFACE on error
	virtual int RCALL GetTypeInfo(int interfaceIndex, int methodID, int paramIndex)=0; // returns UNSUPPORTED_INTERFACE, UNSUPPORTED_METHOD or INVALID_PARAM on error
	virtual int RCALL _GetTypeInfo(int interfaceIndex, int methodID, int paramIndex)=0;	
	virtual int RCALL Invoke(int interfaceIndex, int methodID, void **params, int nparam, void *retval)=0; // returns UNSUPPORTED_INTERFACE or UNSUPPORTED_METHOD on error
	virtual int RCALL _Invoke(int interfaceIndex, int methodID, void **params, int nparam, void *retval)=0;	
	virtual void RCALL OverrideInvoke(RCOMObject *newObject)=0; // this will override "GetTypeInfo" and "Invoke" methods
	virtual bool RCALL SetParent(RCOMObject *parent)=0; // returns false if parent already set
};

struct RCOMString
{
	virtual long RCALL AddRef()=0;
	virtual long RCALL Release()=0;
	virtual const wchar_t* RCALL GetString()=0;
	virtual int RCALL GetLength()=0;
};

struct RCOMSystemAPI
{
	virtual bool RCALL IsFactoryAvailable(const GUID *factID)=0;
	virtual RCOMObject* RCALL GetObjectFromFactory(const GUID *factID,const GUID *clsID)=0;
	virtual bool RCALL ShutdownSystem()=0; // shutdowns the system asynchronously
	virtual wchar_t* RCALL GetApplicationDir()=0; // do not free returned string
	virtual int RCALL GetCommandlineArgCount()=0;
	virtual wchar_t* RCALL GetCommandlineArgument(int index)=0; // do not free returned string
	virtual void RCALL SetNewInstanceListener(RCOMObject *newInstanceListener)=0; // can set only one listener. old one will release.
	virtual RCOMObject* RCALL GetNewInstanceListener()=0; // hook returned object to make chain of NewInstanceListeners!
};

struct RCOMFactory
{
	virtual long RCALL AddRef()=0;
	virtual long RCALL Release()=0;
	virtual const GUID* RCALL GetFactoryID()=0;
	virtual RCOMObject* RCALL QueryObject(const GUID *clsID)=0;

	virtual bool RCALL Init(RCOMSystemAPI *systemAPI,bool isFinal)=0;
	virtual void RCALL Exec()=0;
	virtual void RCALL Stop()=0;
	virtual void RCALL Quit()=0;
};

// {B181404B-4FE3-4c53-BFE9-536ED363D521}
static const GUID IID_INEWINSTANCELISTENER = { 0xb181404b, 0x4fe3, 0x4c53, { 0xbf, 0xe9, 0x53, 0x6e, 0xd3, 0x63, 0xd5, 0x21 } };

interface INewInstanceListener
{
	enum {
    	INEWINSTANCELISTENER_ON_NEW_INSTANCE_START=10,
  	};

	virtual void OnNewInstanceStart(RCOMString **args,int argc)=0;
};

class RCString: public RCOMString
{
protected:
	wchar_t* text;
	int length;
	long refCount;

	~RCString()
	{
		if(text)
			free(text);
	}
	
public:

	RCString(const wchar_t* text, int len)
	{
		if(len)
		{
			this->text=(wchar_t*)malloc((len+1)*sizeof(wchar_t));
			length=len;
			wcscpy(this->text,text); // use wcscpy_s
		}else
		{
			if (text == 0) // null string
			{
				this->text = 0;
				length = 0;
			}
			else // empty string
			{
				this->text = (wchar_t*)malloc(sizeof(wchar_t));
				*this->text = 0;
				length = len;
			}
		}
		refCount=1;
	}

	RCString(const wchar_t* text)
	{
		int len=wcslen(text);
		if(len)
		{
			this->text=(wchar_t*)malloc((len+1)*sizeof(wchar_t));
			length=len;
			wcscpy(this->text,text); // use wcscpy_s
		}else
		{
			if (text == 0) // null string
			{
				this->text = 0;
				length = 0;
			}else // empty string
			{
				this->text = (wchar_t*)malloc(sizeof(wchar_t));
				*this->text = 0;
				length = len;
			}
		}
		refCount=1;
	}	

	long RCALL AddRef()
	{
		return InterlockedIncrement(&refCount);
	}

	long RCALL Release()
	{
	  	long res=InterlockedDecrement(&refCount);
		if(res==0)
			delete this;
		return res;
	}

	const wchar_t* RCALL GetString()
	{
		return text;
	}

	int RCALL GetLength()
	{
		return length;
	}	
};

inline RCOMString* MakeString(const wchar_t* text)
{
	return (RCOMString*)new RCString(text);
}

inline RCOMString* MakeString(const wchar_t* text, int len)
{
	return (RCOMString*)new RCString(text,len);
}

#define DEFINE_RCOM_OBJECT_METHODS \
	long RCALL AddRef(); \
	long RCALL Release(); \
	const GUID* RCALL GetClassID(); \
	int RCALL GetInterfaceIndex(const GUID *interfaceID); \
	int RCALL GetTypeInfo(int interfaceIndex, int methodID, int paramIndex); \
	int RCALL _GetTypeInfo(int interfaceIndex, int methodID, int paramIndex); \
	int RCALL Invoke(int interfaceIndex, int methodID, void **params, int nparam, void *retval); \
	int RCALL _Invoke(int interfaceIndex, int methodID, void **params, int nparam, void *retval); \
	void RCALL OverrideInvoke(RCOMObject *newObject); \
	bool RCALL SetParent(RCOMObject *parent); 


#define ADD_REF_CODE return InterlockedIncrement(&refCount);

#define RELEASE_CODE \
  	long res=InterlockedDecrement(&refCount); \
	if(res==0) \
		delete this; \
	return res;

#define IMPLEMENTS_REFERENCE_COUNT \
	long RC_CLASS::AddRef() \
	{ \
		ADD_REF_CODE \
	} \
	long RC_CLASS::Release() \
	{ \
		RELEASE_CODE \
	}

#define IMPLEMENTS_GET_CLASS_ID(CLSID) \
	const GUID* RC_CLASS::GetClassID() \
	{ \
  		return &CLSID; \
	}

#define IMPLEMENTS_OVERRIDE_INVOKE \
	void RC_CLASS::OverrideInvoke(RCOMObject *newObject) \
	{ \
		if(!overrideObj) \
		{ \
			newObject->SetParent((RCOMObject*)this); \
		}else \
		{ \
			overrideObj->OverrideInvoke(newObject); \
			overrideObj->Release(); \
		} \
		newObject->AddRef(); \
		overrideObj=newObject; \
	}

#define IMPLEMENTS_SET_PARENT \
	bool RC_CLASS::SetParent(RCOMObject *parent) \
	{ \
		if(this->parent) \
			return false; \
 		\
		parent->AddRef(); \
		this->parent=parent; \
		\
		return true; \
	}

#define REDIRECT_GET_TYPE_INFO \
	int RC_CLASS::GetTypeInfo(int interfaceIndex, int methodID, int paramIndex) \
	{ \
		if(!overrideObj) \
			return _GetTypeInfo(interfaceIndex,methodID,paramIndex); \
		return overrideObj->GetTypeInfo(interfaceIndex,methodID,paramIndex); \
	}

#define REDIRECT_INVOKE \
	int RC_CLASS::Invoke(int interfaceIndex, int methodID, void **params, int nparam, void *retval) \
	{ \
		if(!overrideObj) \
			return _Invoke(interfaceIndex,methodID,params,nparam,retval); \
		return overrideObj->Invoke(interfaceIndex,methodID,params,nparam,retval); \
	}

#define DEFINE_RCOM_OBJECT_PROPERTIES \
	long refCount; \
  	RCOMObject *overrideObj; \
  	RCOMObject *parent; 

#define INITIALIZE_RCOM_OBJECT_PROPERTIES \
  	refCount=1; \
  	overrideObj=0; \
  	parent=0;

#define RELEASE_RCOM_OBJECT_PROPERTIES \
	if(overrideObj) \
		overrideObj->Release(); \
	if(parent) \
		parent->Release();


// helper to unpack params	
class DispatchHelper
{
public:
	// original code from WASABI

	template <class CLASSNAME, class RETVAL>
	void cb(RETVAL (CLASSNAME::*fn)(), void *retval, void **params) {
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)();
	}

	template <class CLASSNAME>
	void vcb(void (CLASSNAME::*fn)(), void **params) {
		(static_cast<CLASSNAME *>(this)->*fn)();
	}

	template <class CLASSNAME, class RETVAL, class PARAM1>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1);
	}

	template <class CLASSNAME, class PARAM1>
	void vcb(void (CLASSNAME::*fn)(PARAM1), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1);
	}

	template <class CLASSNAME, class PARAM1, class PARAM2>
	void vcb(void (CLASSNAME::*fn)(PARAM1, PARAM2), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2);
	}

	template <class CLASSNAME, class RETVAL, class PARAM1, class PARAM2>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1, PARAM2), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2);
	}

	// 3 params
	template <class CLASSNAME, class PARAM1, class PARAM2, class PARAM3>
	void vcb(void (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3);
	}

	template <class CLASSNAME, class RETVAL, class PARAM1, class PARAM2, class PARAM3>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3);
	}

	// 4 params
	template <class CLASSNAME, class PARAM1, class PARAM2, class PARAM3, class PARAM4>
	void vcb(void (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4);
	}

	template <class CLASSNAME, class RETVAL, class PARAM1, class PARAM2, class PARAM3, class PARAM4>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4);
	}

	// 5 params
	template <class CLASSNAME, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5>
	void vcb(void (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5);
	}

	template <class CLASSNAME, class RETVAL, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5);
	}

	// 6 params
	template <class CLASSNAME, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6>
	void vcb(void (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6);
	}

	template <class CLASSNAME, class RETVAL, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6);
	}

	// 7 params
	template <class CLASSNAME, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7>
	void vcb(void (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6, PARAM7), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		PARAM7 *p7 = static_cast<PARAM7*>(params[6]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6, *p7);
	}

	template <class CLASSNAME, class RETVAL, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6, PARAM7), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		PARAM7 *p7 = static_cast<PARAM7*>(params[6]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6, *p7);
	}

	// 8 params
	template <class CLASSNAME, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8>
	void vcb(void (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6, PARAM7, PARAM8), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		PARAM7 *p7 = static_cast<PARAM7*>(params[6]);
		PARAM8 *p8 = static_cast<PARAM8*>(params[7]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6, *p7, *p8);
	}

	template <class CLASSNAME, class RETVAL, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6, PARAM7, PARAM8), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		PARAM7 *p7 = static_cast<PARAM7*>(params[6]);
		PARAM8 *p8 = static_cast<PARAM8*>(params[7]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6, *p7, *p8);
	}

	// 9 params
	template <class CLASSNAME, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8, class PARAM9>
	void vcb(void (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6, PARAM7, PARAM8, PARAM9), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		PARAM7 *p7 = static_cast<PARAM7*>(params[6]);
		PARAM8 *p8 = static_cast<PARAM8*>(params[7]);
		PARAM9 *p9 = static_cast<PARAM9*>(params[8]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6, *p7, *p8, *p9);
	}

	template <class CLASSNAME, class RETVAL, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8, class PARAM9>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6, PARAM7, PARAM8, PARAM9), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		PARAM7 *p7 = static_cast<PARAM7*>(params[6]);
		PARAM8 *p8 = static_cast<PARAM8*>(params[7]);
		PARAM9 *p9 = static_cast<PARAM9*>(params[8]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6, *p7, *p8, *p9);
	}

	// 10 params
	template <class CLASSNAME, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8, class PARAM9, class PARAM10>
	void vcb(void (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6, PARAM7, PARAM8, PARAM9, PARAM10), void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		PARAM7 *p7 = static_cast<PARAM7*>(params[6]);
		PARAM8 *p8 = static_cast<PARAM8*>(params[7]);
		PARAM9 *p9 = static_cast<PARAM9*>(params[8]);
		PARAM10 *p10 = static_cast<PARAM10*>(params[9]);
		(static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6, *p7, *p8, *p9, *p10);
	}

	template <class CLASSNAME, class RETVAL, class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8, class PARAM9, class PARAM10>
	void cb(RETVAL (CLASSNAME::*fn)(PARAM1, PARAM2, PARAM3, PARAM4, PARAM5, PARAM6, PARAM7, PARAM8, PARAM9, PARAM10), void *retval, void **params) {
		PARAM1 *p1 = static_cast<PARAM1*>(params[0]);
		PARAM2 *p2 = static_cast<PARAM2*>(params[1]);
		PARAM3 *p3 = static_cast<PARAM3*>(params[2]);
		PARAM4 *p4 = static_cast<PARAM4*>(params[3]);
		PARAM5 *p5 = static_cast<PARAM5*>(params[4]);
		PARAM6 *p6 = static_cast<PARAM6*>(params[5]);
		PARAM7 *p7 = static_cast<PARAM7*>(params[6]);
		PARAM8 *p8 = static_cast<PARAM8*>(params[7]);
		PARAM9 *p9 = static_cast<PARAM9*>(params[8]);
		PARAM10 *p10 = static_cast<PARAM10*>(params[9]);
		*static_cast<RETVAL*>(retval) = (static_cast<CLASSNAME *>(this)->*fn)(*p1, *p2, *p3, *p4, *p5, *p6, *p7, *p8, *p9, *p10);
	}
};

// helper to pack params	
class ClientHelper
{
public:

	// original code from WASABI

	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID) {
		return obj->Invoke(interfaceIndex, methodID, 0,0,NULL);
	}

	template<class PARAM1>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1) {
		void *params[1] = { &param1 };
		return obj->Invoke(interfaceIndex, methodID, params, 1, NULL);
	}

	template<class PARAM1, class PARAM2>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2) {
		void *params[2] = { &param1, &param2 };
		return obj->Invoke(interfaceIndex, methodID, params, 2, NULL);
	}

	template<class PARAM1, class PARAM2, class PARAM3>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3) {
		void *params[3] = { &param1, &param2, &param3 };
		return obj->Invoke(interfaceIndex, methodID, params, 3, NULL);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4) {
		void *params[4] = { &param1, &param2, &param3, &param4 };
		return obj->Invoke(interfaceIndex, methodID, params, 4, NULL);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5) {
		void *params[5] = { &param1, &param2, &param3, &param4, &param5 };
		return obj->Invoke(interfaceIndex, methodID, params, 5, NULL);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6) {
		void *params[6] = { &param1, &param2, &param3, &param4, &param5, &param6 };
		return obj->Invoke(interfaceIndex, methodID, params, 6, NULL);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6, PARAM7 param7) {
		void *params[7] = { &param1, &param2, &param3, &param4, &param5, &param6 , &param7 };
		return obj->Invoke(interfaceIndex, methodID, params, 7, NULL);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6, PARAM7 param7, PARAM8 param8) {
		void *params[8] = { &param1, &param2, &param3, &param4, &param5, &param6 , &param7 , &param8 };
		return obj->Invoke(interfaceIndex, methodID, params, 8, NULL);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8, class PARAM9>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6, PARAM7 param7, PARAM8 param8, PARAM9 param9) {
		void *params[9] = { &param1, &param2, &param3, &param4, &param5, &param6 , &param7 , &param8 , &param9 };
		return obj->Invoke(interfaceIndex, methodID, params, 9, NULL);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8, class PARAM9, class PARAM10>
	int _voidcall(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6, PARAM7 param7, PARAM8 param8, PARAM9 param9, PARAM10 param10) {
		void *params[10] = { &param1, &param2, &param3, &param4, &param5, &param6 , &param7 , &param8 , &param9 , &param10 };
		return obj->Invoke(interfaceIndex, methodID, params, 10, NULL);
	}




	int _call(RCOMObject *obj, int interfaceIndex, int methodID, void *retval) {
		return obj->Invoke(interfaceIndex, methodID, 0, 0, retval);
	}

	template<class PARAM1>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, void *retval) {
		void *params[1] = { &param1 };
		return obj->Invoke(interfaceIndex, methodID, params, 1, retval);
	}

	template<class PARAM1, class PARAM2>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, void *retval) {
		void *params[2] = { &param1, &param2 };
		return obj->Invoke(interfaceIndex, methodID, params, 2, retval);
	}

	template<class PARAM1, class PARAM2, class PARAM3>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, void *retval) {
		void *params[3] = { &param1, &param2, &param3 };
		return obj->Invoke(interfaceIndex, methodID, params, 3, retval);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, void *retval) {
		void *params[4] = { &param1, &param2, &param3, &param4 };
		return obj->Invoke(interfaceIndex, methodID, params, 4, retval);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, void *retval) {
		void *params[5] = { &param1, &param2, &param3, &param4, &param5 };
		return obj->Invoke(interfaceIndex, methodID, params, 5, retval);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6, void *retval) {
		void *params[6] = { &param1, &param2, &param3, &param4, &param5, &param6 };
		return obj->Invoke(interfaceIndex, methodID, params, 6, retval);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6, PARAM7 param7, void *retval) {
		void *params[7] = { &param1, &param2, &param3, &param4, &param5, &param6, &param7 };
		return obj->Invoke(interfaceIndex, methodID, params, 7, retval);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6, PARAM7 param7, PARAM8 param8, void *retval) {
		void *params[8] = { &param1, &param2, &param3, &param4, &param5, &param6, &param7, &param8 };
		return obj->Invoke(interfaceIndex, methodID, params, 8, retval);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8, class PARAM9>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6, PARAM7 param7, PARAM8 param8, PARAM9 param9, void *retval) {
		void *params[9] = { &param1, &param2, &param3, &param4, &param5, &param6, &param7, &param8, &param9 };
		return obj->Invoke(interfaceIndex, methodID, params, 9, retval);
	}

	template<class PARAM1, class PARAM2, class PARAM3, class PARAM4, class PARAM5, class PARAM6, class PARAM7, class PARAM8, class PARAM9, class PARAM10>
	int _call(RCOMObject *obj, int interfaceIndex, int methodID, PARAM1 param1, PARAM2 param2, PARAM3 param3, PARAM4 param4, PARAM5 param5, PARAM6 param6, PARAM7 param7, PARAM8 param8, PARAM9 param9, PARAM10 param10, void *retval) {
		void *params[10] = { &param1, &param2, &param3, &param4, &param5, &param6, &param7, &param8, &param9, &param10 };
		return obj->Invoke(interfaceIndex, methodID, params, 10, retval);
	}
};


#endif