
#include "SystemAPI.h"
#include "RCOMLoader.h"

#ifdef ENABLE_GTK
	#include <gtk/gtk.h>
#endif

#ifdef WIN32 // windows
	#include <windows.h>
#else // linux
	#include <unistd.h>
#endif

char appDir[512];

SystemAPI::SystemAPI(int argc, char* argv[], RCOMLoader *loader)
{
	this->loader = loader;
	newInstanceListener = 0;
	shutdownCalled = false;
	factoryList = new KPointerList<FactoryHolder*>();

#ifdef WIN32
	GetModuleFileNameA(GetModuleHandleA(0), appDir, 512);

	char *p;
	for (p = appDir; *p; p++) { }	// find end
	for (; p > appDir && *p != '\\'; p--) { } // back up to last backslash
	*p = 0;	// kill it
#else
	int size = readlink("/proc/self/exe", appDir, 512 - 1);
	appDir[size] = 0;

	char *p;
	for (p = appDir; *p; p++) { }	// find end
	for (; p > appDir && *p != '/'; p--) { } // back up to last backslash
	*p = 0;	// kill it
#endif

	nNumArgs = argc;
	args = argv;	
}

SystemAPI::~SystemAPI()
{
	if(newInstanceListener)
		delete newInstanceListener;

	// delete factory holder objects
	for(int i=0; i < factoryList->GetSize(); i++)
		delete factoryList->GetPointer(i);

	delete factoryList;
}

void SystemAPI::AddFactory(RCOMFactory *factory, bool runComponent)
{
	FactoryHolder *holder = new FactoryHolder();
	holder->factory = factory;
	holder->initOK = false;

	factoryList->AddPointer(holder);

	if(runComponent)
	{
		holder->initOK = factory->Init((RCOMSystemAPI*)this, false);
		factory->Exec();
	}
}

void SystemAPI::SetNewInstanceListener(RCOMObject *newInstanceListener)
{
	if(newInstanceListener)
	{
		if(newInstanceListener->GetInterfaceIndex(&IID_INEWINSTANCELISTENER)!=UNSUPPORTED_INTERFACE)
		{
			if(this->newInstanceListener) // remove old one
				delete this->newInstanceListener;

			this->newInstanceListener = new PNewInstanceListener(newInstanceListener);
		}
	}
}

RCOMObject* SystemAPI::GetNewInstanceListener()
{
	if(this->newInstanceListener)
	{
		newInstanceListener->obj->AddRef();
		return newInstanceListener->obj;
	}
	return 0;
}

bool SystemAPI::LoadComponent(char* path)
{
	return loader->LoadComponent(path, true);
}

void SystemAPI::InitAll()
{

	int prevSuccessCount=0;

	while(true)
	{
		int initCalledCount=0;
		int successCount=0;
		for(int i=0;i<factoryList->GetSize();i++)
		{
			FactoryHolder *holder=factoryList->GetPointer(i);
			if(!holder->initOK)
			{
				bool retval = holder->factory->Init((RCOMSystemAPI*)this,false);
				holder->initOK=retval;

				initCalledCount++;
				if(retval)
					successCount++;
			}
		}

		if(initCalledCount==successCount) // everything ok
		{
			break;
		}else if(prevSuccessCount==successCount) // circular or missing dependancies
		{
			// inform it to factory by setting "isFinal" flag
			// return true at "Init" if factory can continue with missing dependencies
			// your factory will not be available to others if you return false
			// other methods like "Exec","Stop","Quit" will be called regardles of the return value
			for(int i=0;i<factoryList->GetSize();i++)
			{
				FactoryHolder *holder=factoryList->GetPointer(i);
				if(!holder->initOK)
				{
					bool retval = holder->factory->Init((RCOMSystemAPI*)this,true);
					holder->initOK=retval;
				}
			}
			break;
		}

		prevSuccessCount=successCount;
	}
}

void SystemAPI::ExecAll()
{
	for(int i=0;i<factoryList->GetSize();i++)
	{
		FactoryHolder *holder=factoryList->GetPointer(i);
		holder->factory->Exec();
	}
}

void SystemAPI::StopAll()
{
	for(int i=0;i<factoryList->GetSize();i++)
	{
		FactoryHolder *holder=factoryList->GetPointer(i);
		holder->factory->Stop();
	}
}

void SystemAPI::QuitAll()
{
	for(int i=0;i<factoryList->GetSize();i++)
	{
		FactoryHolder *holder=factoryList->GetPointer(i);
		holder->factory->Quit();
	}
}

void SystemAPI::ReleaseAllFactories()
{
	// release all factories
	for(int i=0;i<factoryList->GetSize();i++)
	{
		factoryList->GetPointer(i)->factory->Release();
	}
}

bool SystemAPI::IsFactoryAvailable(const GUID *factID)
{
	for(int i=0;i<factoryList->GetSize();i++)
	{
		FactoryHolder *holder=factoryList->GetPointer(i);
		RCOMFactory *factory=holder->factory;

		const GUID *guid=factory->GetFactoryID();
		if(IsEqualGUID(*factID,*guid))
		{
			if(holder->initOK)
				return true;
			return false;
		}
	}
	return false;
}

RCOMObject* SystemAPI::GetObjectFromFactory(const GUID *factID,const GUID *clsID)
{
	for(int i=0;i<factoryList->GetSize();i++)
	{
		FactoryHolder *holder=factoryList->GetPointer(i);
		RCOMFactory *factory=holder->factory;

		const GUID *guid=factory->GetFactoryID();
		if(IsEqualGUID(*factID,*guid))
		{
			if(holder->initOK)
			{
				return holder->factory->QueryObject(clsID);
			}
			return 0;
		}
	}
	return 0;	
}

char* SystemAPI::GetApplicationDir()
{
	return appDir;
}

int SystemAPI::GetCommandlineArgCount()
{
	return nNumArgs-1;
}

char* SystemAPI::GetCommandlineArgument(int index)
{
	if(index <nNumArgs)
		return args[index+1];
	return 0;
}

bool SystemAPI::ShutdownSystem()
{
	if(shutdownCalled)
		return false;

	shutdownCalled=true;

#ifdef ENABLE_GTK
	gtk_main_quit ();
#endif

	return true;
}
