
// portable implementation (c) 2018 R.Hasaranga

/*
    RFC - KPointerList.h
    Copyright (C) 2010 Ruchira Hasaranga
  
    This software is provided 'as-is', without any express or implied
    warranty.  In no event will the authors be held liable for any damages
    arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute it
    freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you must not
       claim that you wrote the original software. If you use this software
       in a product, an acknowledgment in the product documentation would be
       appreciated but is not required.
    2. Altered source versions must be plainly marked as such, and must not be
       misrepresented as being the original software.
    3. This notice may not be removed or altered from any source distribution.
      
*/


#ifndef _RFC_KPOINTERLIST_H_
#define _RFC_KPOINTERLIST_H_

#ifdef WIN32 // windows

	#include <windows.h>

	#define PORTABLE_CRITICAL_SECTION CRITICAL_SECTION 

	#define INIT_CRITICAL_SECTION InitializeCriticalSection(&criticalSection);	

	#define ENTER_CRITICAL_SECTION EnterCriticalSection(&criticalSection);

	#define LEAVE_CRITICAL_SECTION LeaveCriticalSection(&criticalSection);

	#define DELETE_CRITICAL_SECTION DeleteCriticalSection(&criticalSection);

#else // linux

	#include <pthread.h>

	#define PORTABLE_CRITICAL_SECTION pthread_mutex_t

	#define INIT_CRITICAL_SECTION pthread_mutexattr_t mAttr; \
	pthread_mutexattr_settype(&mAttr, PTHREAD_MUTEX_RECURSIVE_NP); \
	pthread_mutex_init(&criticalSection, &mAttr); \
	pthread_mutexattr_destroy(&mAttr);

	#define ENTER_CRITICAL_SECTION pthread_mutex_lock(&criticalSection);

	#define LEAVE_CRITICAL_SECTION pthread_mutex_unlock(&criticalSection);

	#define DELETE_CRITICAL_SECTION pthread_mutex_destroy (&criticalSection);

#endif

#define PTR_SIZE sizeof(void*)

#include <malloc.h>

/**
	Holds a resizable list of pointers.
	To make all the methods thread-safe, pass true for isThreadSafe parameter of constructor.
	Duplicated items allowed!
	index is between 0 to (item count-1)

	e.g. @code
	KButton btn1;
	KPointerList<KButton*> btnList(1024,true);
	btnList.AddPointer(&btn1);
	btnList.AddPointer(&btn1);
	@endcode
*/
template<class T>
class KPointerList
{
protected:
	int size;
	int roomCount;
	int roomIncrement;
	void* list;

	PORTABLE_CRITICAL_SECTION criticalSection;
	bool isThreadSafe;

public:
	/**
		Constructs PointerList object.
		@param roomIncrement initial and reallocation size of internal memory block in DWORDS
		@param isThreadSafe make all the methods thread-safe
	*/
	KPointerList(int roomIncrement=1024, bool isThreadSafe=false) // 1024*4=4096 = default alignment!
	{
		roomCount = roomIncrement;
		this->roomIncrement = roomIncrement;
		this->isThreadSafe = isThreadSafe;
		size = 0;
		list = malloc(roomCount * PTR_SIZE); // 1 DWORD = 4 bytes!
		
		if(isThreadSafe)
		{
			INIT_CRITICAL_SECTION
		}
	}

	/**
		Adds new item to the list.
		@returns false if memory allocation failed!
	*/
	bool AddPointer(T pointer)
	{
		if(isThreadSafe)
		{
			ENTER_CRITICAL_SECTION // thread safe!
		}

		if(roomCount>=(size+1)) // no need reallocation. coz room count is enough!
		{
			*(void**)(list + (size * PTR_SIZE)) = (void*)pointer;

			size++;
			if(isThreadSafe)
			{
				LEAVE_CRITICAL_SECTION
			}
			return true;
		}else // require reallocation!
		{
			roomCount += roomIncrement;
			void* retVal = realloc(list, roomCount * PTR_SIZE);
			if(retVal)
			{
				list = retVal;
				*(void**)(list + (size * PTR_SIZE)) = (void*)pointer;

				size++;
				if(isThreadSafe)
				{
					LEAVE_CRITICAL_SECTION
				}
				return true;
			}else // memory allocation failed!
			{
				if(isThreadSafe)
				{
					LEAVE_CRITICAL_SECTION
				}
				return false;
			}
		}
	}

	/**
		Get pointer at id.
		@returns 0 if id is out of range!
	*/
	T GetPointer(int id)
	{
		if(isThreadSafe)
		{
			ENTER_CRITICAL_SECTION
		}

		if( (0 <= id) & (id < size) ) // checks for valid range!
		{	
			T object = (T) (*(void**)(list + (id * PTR_SIZE)));
			if(isThreadSafe)
			{
				LEAVE_CRITICAL_SECTION
			}
			return object;
		}else // out of range!
		{
			if(isThreadSafe)
			{
				LEAVE_CRITICAL_SECTION
			}
			return 0;
		}
	}

	/**
		Replace pointer of given id with new pointer
		@returns false if id is out of range!
	*/
	bool SetPointer(int id, T pointer)
	{
		if(isThreadSafe)
		{
			ENTER_CRITICAL_SECTION
		}

		if( (0 <= id) & (id < size) )
		{			
			*(void**)(list + (id * PTR_SIZE)) = (void*)pointer;
			if(isThreadSafe)
			{
				LEAVE_CRITICAL_SECTION
			}
			return true;
		}else // out of range!
		{
			if(isThreadSafe)
			{
				LEAVE_CRITICAL_SECTION
			}
			return false;
		}
	}

	/**
		Remove pointer of given id
		@returns false if id is out of range!
	*/
	bool RemovePointer(int id)
	{
		if(isThreadSafe)
		{
			ENTER_CRITICAL_SECTION
		}

		if( (0 <= id) & (id < size) )
		{	
			int newRoomCount = (((size - 1) / roomIncrement) + 1) * roomIncrement;
			void* newList = malloc(newRoomCount * PTR_SIZE);

			for(register int i=0,j=0; i < size; i++)
			{
				if(i != id)
				{
					*(void**)(newList + (j * PTR_SIZE)) = *(void**)(list + (i * PTR_SIZE));
					j++;
				}	
			}

			free(list); // free old list!
			list = newList;
			roomCount = newRoomCount;
			size--;

			if(isThreadSafe)
			{
				LEAVE_CRITICAL_SECTION
			}
			return true;

		}else // out of range!
		{
			if(isThreadSafe)
			{
				LEAVE_CRITICAL_SECTION
			}
			return false;
		}

	}

	/**
		Clears the list!
	*/
	void RemoveAll()// remove all pointers from list!
	{
		if(isThreadSafe)
		{
			ENTER_CRITICAL_SECTION
		}

		free(list);
		roomCount = roomIncrement;
		list = malloc(roomCount * PTR_SIZE);
		size = 0;

		if(isThreadSafe)
		{
			LEAVE_CRITICAL_SECTION
		}
	}

	/**
		Call destructors of all objects which are pointed by pointers in the list.
		Also clears the list. Never use pointerlist again if you call this method
		with reAllocate set to false.
	*/
	void DeleteAll(bool reAllocate = true)
	{
		if(isThreadSafe)
		{
			ENTER_CRITICAL_SECTION
		}

		for(int i=0; i < size; i++)
		{
			T object=(T) (*(void**)(list + (i * PTR_SIZE)));
			delete object;
		}

		if(reAllocate)
		{
			free(list);
			roomCount = roomIncrement;
			list = malloc(roomCount * PTR_SIZE);
			size = 0;
		}

		if(isThreadSafe)
		{
			LEAVE_CRITICAL_SECTION
		}
	}

	/**
		Finds the id of the first pointer which matches the pointer passed in.
		@returns -1 if not found!
	*/
	int GetID(T pointer)
	{
		if(isThreadSafe)
		{
			ENTER_CRITICAL_SECTION
		}
		for(register int i=0; i < size; i++)
		{
			if(*(void**)(list + (i * PTR_SIZE)) == (void*)pointer)
			{
				if(isThreadSafe)
				{
					LEAVE_CRITICAL_SECTION
				}
				return i;
			}
		}
		if(isThreadSafe)
		{
			LEAVE_CRITICAL_SECTION
		}
		return -1;
	}

	/**
		@returns item count in the list
	*/
	int GetSize()
	{
		return size;
	}

	/** Destructs PointerList object.*/
	~KPointerList()
	{
		free(list);
		if(isThreadSafe)
		{
			DELETE_CRITICAL_SECTION
		}
	}

};

#endif
